from os.path import basename
from urllib.parse import ParseResult
from .extractor import Extractor


class URL:

    def __init__(self, url: str) -> None:
        self.url = url.strip(' .')
        self.extractor = Extractor.getExtractor(self.url)

        self.all = self.extractor.all

    @property
    def domain_name(self) -> str:
        return self.extractor.domain_name

    @property
    def normalized_domain_name(self) -> str:
        """
        Abridged domain name, stripping uninteresting patterns.
        Used to keep login variations to a minimum for database efficiency.
        """
        return self.extractor.normalize_domain_name

    @property
    def normalized_hostname(self) -> str:
        """
        Abridged hostname, stripping uninteresting patterns.
        Used to keep login variations to a minimum for database efficiency.
        """
        return self.extractor.normalized_hostname

    @property
    def default_mimetype(self) -> str:
        return self.extractor.default_mimetype

    @property
    def hostname(self) -> str:
        return self.extractor.hostname

    @property
    def parse_url(self) -> ParseResult:
        return self.extractor.parse_url

    @property
    def query_string(self) -> str:
        return self.parse_url.query

    @property
    def search_term(self) -> str:
        return self.extractor.search_term

    @property
    def title(self) -> str:
        return self.extractor.title

    @property
    def is_tech_service(self) -> bool:
        return self.extractor.is_tech_service

    @property
    def filename(self) -> str:
        return basename(self.parse_url.path)

    def __str__(self) -> str:
        return self.url

    def __repr__(self) -> str:
        return f'{self.__class__.__name__}({self.url})'

    def fetch(self):
        raise NotImplementedError('Retrieve URL body from web server or file')


__all__ = ('URL',)
