import unittest
from ..mimetypes import Mimetype
from ..mimes import REQ_CODES

MIMES = {
    'bmp': 'image/bmp',
    'document': 'text/html',
    'application/html': 'text/html',
    'application/dns-udpwireformat': 'application/dns',
    'application/dns-json': 'application/dns',
    'application/dns+json': 'application/dns',
    'html': 'text/html',
    'xml': 'application/xml',
    '*/*': 'text/html',
    'smartfinddata': 'application/json',
    '$picmime': 'image/jpg',
    'binary': 'application/binary',
    'image/': 'image/jpg',
    'image': 'image/jpg',
    'jpg': 'image/jpg',
    'jpeg': 'image/jpg',
    'js': 'application/javascript',
    'javascript': 'application/javascript',
    'ecmascript': 'application/javascript',
    'captcha.jpg': 'image/jpg',
    'captcha.jpeg': 'image/jpg',
    'file': 'file/file',
    'gif': 'image/gif',
    'tif': 'image/tiff',
    'tiff': 'image/tiff',
    'pdf': 'application/pdf',
    'png': 'image/png',
    'video': 'video/mp4',
    'woff': 'application/woff',
    'ttf': 'application/font',
    'otf': 'application/font',
    'application/font-opentype': 'application/font',
    'utf-8': 'text/plain',
    'utf8': 'text/plain',
    'text': 'text/plain',
    'text/text': 'text/plain',
    'type/js': 'application/javascript',
    'text/javascript': 'application/javascript',
    'text/x-javascript': 'application/javascript',
    '[image/jpeg]': 'image/jpg',
    'application/soap+x': 'application/soap',
    'application/dssc+der': 'application/dssc',
    'application/x-dtbook+xml': 'application/dtbook',
    'application/vnd.las.las+xml': 'application/las.las',
    'application/postscript; ghostview %s': 'application/postscript',
    'image/image/jpg': 'image/jpg',
    'application/application/application/json': 'application/json',
    'application/x-java-class': 'application/java',
    'application/x-java-bytecode': 'application/java',
    'application/x-java-byte-code': 'application/java',
    'application/x-protobuf-gz': 'application/protobuf',
    'text/proto': 'application/protobuf',
    'x-world/x-3dmf': 'world/3dmf',
    'text/vnd.abc': 'text/abc',
    'text/.css': 'text/css',
    'text.css': 'text/css',
    'text/x-script.perl': 'text/perl',
    'text/x-perl': 'text/perl',
    'text/x-script.perl-module': 'text/perl',
    'text/css; charset=utf8': 'text/css',
    'audio/x-aiff': 'audio/aiff',
    'audio;x-aiff': 'audio/aiff',
    'image/jpg': 'image/jpg',
    'image/jpeg': 'image/jpg',
    'imagejpeg': 'image/jpg',
    'image\jpeg': 'image/jpg',
    'image\\jpeg': 'image/jpg',
    'image;jpeg': 'image/jpg',
    'image/svg+x': 'image/svg',
    'image/svg+xml': 'image/svg',
    'image/x-icon': 'image/icon',
    'image/png,image/gif': 'image/png',
    'image/png, image/gif,image/jpg': 'image/png',
    'image,gif': 'image/gif',
    'text/x-asm': 'text/asm',
    'plain/text': 'text/plain',
    'application/text': 'text/plain',
    'application/hal+json': 'application/json',
    'application/amz-json-1.0': 'application/json',
    'application/amz-json-1.1': 'application/json',
    'application/font-woff; charset=binary': 'application/font',
    'application/x-javascript:; charset=utf-8': 'application/javascript',
    'application/vnd.ms-cab-compressed': 'application/cab-compressed',
    'video/mpeg; xmpeg %s': 'video/mpeg',
    'x-image/vnd.dwg': 'image/dwg',
    'x-image/x-png': 'image/png',
    'whoever/heard': 'whoever/heard',
    'whoever/prs.heard': 'whoever/heard',
    'application/x.mix-transfer': 'application/mix-transfer',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document': 'application/document',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.template': 'application/document',
    'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet': 'application/spreadsheet',
    'application/vnd.openxmlformats-officedocument.spreadsheetml.template': 'application/spreadsheet',
    'application/vnd.openxmlformats-officedocument.presentationml.presentation': 'application/presentation',
    'application/vnd.openxmlformats-officedocument.presentationml.template': 'application/presentation',
}

MIME_CODE_MAP = {
    'text/html': REQ_CODES['page'],
    'image/svg': REQ_CODES['pagevisuals'],
    'text/css': REQ_CODES['pageassets'],
    'application/hessian': REQ_CODES['pageassets'],
    'application/json': REQ_CODES['pageassets'],
    'application/protobuf': REQ_CODES['pageassets'],
    'application/protobuffer': REQ_CODES['pageassets'],
    'text/plain': REQ_CODES['files'],
    'application/binary': REQ_CODES['files'],
    'application/octet-stream': REQ_CODES['files'],
    'application/octet-stream-compressible': REQ_CODES['files'],
    'application/google.octet-stream-compressible': REQ_CODES['files'],
    'binary/octet-stream': REQ_CODES['files'],
    'application/gzip': REQ_CODES['programs'],
    'application/exe': REQ_CODES['programs'],
    'application/api': REQ_CODES['api'],
    'application/smil': REQ_CODES['api'],
    'application/sync': REQ_CODES['api'],
    'application/plist': REQ_CODES['api'],
    'application/rtf': REQ_CODES['documents'],
    'application/pdf': REQ_CODES['books'],
    'application/rpm': REQ_CODES['programs'],
    'audio/mp3': REQ_CODES['audiovideo'],
    'video/flv': REQ_CODES['audiovideo'],
}


class TestExtractors(unittest.TestCase):

    def test_mime(self):
        for mime, clean in MIMES.items():
            fixed = Mimetype(mime).clean
            self.assertEqual(
                clean,
                fixed,
                msg=f'Error cleaning "{mime}". '
                f'Cleaned value "{fixed}" does not equal expected value "{clean}"'
            )

    def test_request_codes(self):
        for mimetype, req_code in MIME_CODE_MAP.items():
            hc = Mimetype(mimetype).hit_code
            self.assertEqual(
                first=hc, second=req_code, msg=f'{mimetype} hit code {hc} != {req_code}'
            )

    def test_cache_read_correctly(self):
        """
        Ensure that attributes are set correctly when loading from cache
        """

        self.assertEqual(Mimetype('image/jpg').hit_code, Mimetype('image/jpg').hit_code)
