import json
from typing import Tuple
import unittest

from uraltimber import URL, Extractor
from ..settings import ROOT


class TestExtractors(unittest.TestCase):

    fixtures: Tuple[str, ...] = ()

    @classmethod
    def setUpClass(cls):
        cls.data = []
        for fixture_file in cls.fixtures:
            with open(f'{ROOT}/tests/{fixture_file}', 'r') as ff:
                cls.data.extend(json.loads(ff.read()))

    def test_fixtures(self):
        for udata in self.data:
            url = URL(udata['url']).all()

            for key in ('domain_name', 'hostname', 'title', 'search_term', 'extension'):
                fixture_val = udata.get(key, '')
                parsed_val = url.get(key, '')
                self.assertEqual(
                    fixture_val,
                    parsed_val,
                    msg=f"Test failed for {udata['url']}. For \"{key}\" key, parsed "
                    f"value \"{parsed_val}\" does not match fixture value \"{fixture_val}\"."
                )


class TestMimeTypes(TestExtractors):
    fixtures = ('fixtures/default_mimetypes.json',)

    def test_fixtures(self):
        for udata in self.data:
            default_mime = URL(udata['url']).default_mimetype
            self.assertEqual(
                udata['mimetype'],
                default_mime,
                msg=f"Test failed for {udata['url']}. Parsed value \"{default_mime}\" does "
                f"not equal expected value \"{udata['mimetype']}\""
            )


class TestGenericURLs(TestExtractors):
    fixtures = ('fixtures/generic.json',)

    def test_generic_extractor(self):
        for udata in self.data:
            self.assertIsInstance(Extractor.getExtractor(udata['url']), Extractor)


class TestAiSearch(TestExtractors):
    fixtures = ('fixtures/ai_search.json',)


class TestEcommerce(TestExtractors):
    fixtures = ('fixtures/ecommerce.json',)


class TestMaps(TestExtractors):
    fixtures = ('fixtures/maps.json',)


class TestMedia(TestExtractors):
    fixtures = ('fixtures/media.json',)

    def test_playing_media_id(self):
        youtube = 'https://www.youtube.com/api/stats/watchtime?ns=yt&el=embedded&cpn=mGpwsBNeUiU3bk3L&ver=2&cmt=72.154&fmt=247&fs=0&rt=186.004&euri=https%3A%2F%2F9to5mac.com%2F&lact=72592&cl=674975304&state=playing&volume=100&cbrand=apple&cbr=Chrome&cbrver=128.0.0.0&c=WEB_EMBEDDED_PLAYER&cver=1.20240915.00.00&cplayer=UNIPLAYER&cos=Macintosh&cosver=10_15_7&cplatform=DESKTOP&hl=en_US&cr=US&len=1489.281&rtn=226&feature=oembed&afmt=251&idpj=0&ldpj=-38&rti=186&size=786%3A443%3A2&inview=1&st=32.153&et=72.154&muted=0&docid=QqTSnI0bfFk&ei=uNbqZrbbIc210_wPwaGZ8AM&plid=AAYiZNrh4T60uJGt&referrer=https%3A%2F%2Fwww.youtube.com%2Fembed%2FQqTSnI0bfFk%3Ffeature%3Doembed%26rel%3D0%26enablejsapi%3D1&of=nJiyl6tUJbE7C7O6bu7jVg'
        extractor = Extractor.getExtractor(youtube)
        self.assertEqual(extractor.playing_media_id(), 'QqTSnI0bfFk')

        vimeo = 'https://player.vimeo.com/progressive_redirect/playback/855791175/rendition/720p/file.mp4?loc=external&log_user=0&signature=712d9d8649842d099b05824810f62a2b81391384660770fb8792f937d5a32c62'
        extractor = Extractor.getExtractor(vimeo)
        self.assertEqual(extractor.playing_media_id(), '855791175')


class TestMusic(TestExtractors):
    fixtures = ('fixtures/music.json',)


class TestPhotography(TestExtractors):
    fixtures = ('fixtures/photography.json',)


class TestPrintMedia(TestExtractors):
    fixtures = ('fixtures/print_media.json',)


class TestResearch(TestExtractors):
    fixtures = ('fixtures/research.json',)


class TestSearchEngines(TestExtractors):
    fixtures = ('fixtures/search_engines.json',)


class TestSocialMedia(TestExtractors):
    fixtures = ('fixtures/social_media.json',)


class TestTranslation(TestExtractors):
    fixtures = ('fixtures/translation.json',)
