from uraltimber.extractor import Extractor
from ..settings import re_icompile

VIMEO_PLAYING_URL = re_icompile(r'vimeo.com/progressive_redirect/playback/(\d+)/')
VIMEO_MEDIA_ID = re_icompile(r'vimeo.com/(?:video/|progressive_redirect/playback/)?(\d+)(/|$)')


class Ustream(Extractor):
    domain_name = 'ustream.tv'
    search_keys = ('q',)


class Vimeo(Extractor):
    domain_name = 'vimeo.com'
    search_keys = ('q',)
    base_media_download_url = 'https://player.vimeo.com/video/{media_id}'

    def media_id(self):
        """
        https://vimeo.com/348257642
        https://player.vimeo.com/video/<media_id>
        https://player.vimeo.com/progressive_redirect/playback/<media_id>/rendition/720p/file.mp4?
        """
        if matched := VIMEO_MEDIA_ID.search(self.url):
            return matched.group(1)
        return ''

    def playing_media_id(self) -> str:
        """
        Vimeo-hosted videos that were played generated a "progressive_redirect" URL.

        https://player.vimeo.com/progressive_redirect/playback/855791175/rendition/720p/file.mp4?loc=external&log_user=0&signature=712d9d8649842d099b05824810f62a2b81391384660770fb8792f937d5a32c62
        """
        if matched := VIMEO_PLAYING_URL.search(self.url):
            return matched.group(1)
        return ''


class YouTube(Extractor):
    domain_name = 'youtube.com'
    search_keys = ('search_query',)
    base_media_download_url = 'https://www.youtube.com/watch?v={media_id}'

    def media_id(self):
        """
        https://www.youtube.com/watch?v=nhe8CnteS14
        https://www.youtube.com/api/stats/watchtime?ns=yt&el=embedded&cpn=mGpwsBNeUiU3bk3L&ver=2&cmt=72.154&fmt=247&fs=0&rt=186.004&euri=https%3A%2F%2F9to5mac.com%2F&lact=72592&cl=674975304&state=playing&volume=100&cbrand=apple&cbr=Chrome&cbrver=128.0.0.0&c=WEB_EMBEDDED_PLAYER&cver=1.20240915.00.00&cplayer=UNIPLAYER&cos=Macintosh&cosver=10_15_7&cplatform=DESKTOP&hl=en_US&cr=US&len=1489.281&rtn=226&feature=oembed&afmt=251&idpj=0&ldpj=-38&rti=186&size=786%3A443%3A2&inview=1&st=32.153&et=72.154&muted=0&docid=QqTSnI0bfFk&ei=uNbqZrbbIc210_wPwaGZ8AM&plid=AAYiZNrh4T60uJGt&referrer=https%3A%2F%2Fwww.youtube.com%2Fembed%2FQqTSnI0bfFk%3Ffeature%3Doembed%26rel%3D0%26enablejsapi%3D1&of=nJiyl6tUJbE7C7O6bu7jVg
        """
        qp = self.query_params
        return qp.get('v') or qp.get('docid')

    def playing_media_id(self) -> str:
        """
        Playing / Paused YouTube URLs look like so:

        https://www.youtube.com/api/stats/watchtime?ns=yt&el=embedded&cpn=mGpwsBNeUiU3bk3L&ver=2&cmt=72.154&fmt=247&fs=0&rt=186.004&euri=https%3A%2F%2F9to5mac.com%2F&lact=72592&cl=674975304&state=playing&volume=100&cbrand=apple&cbr=Chrome&cbrver=128.0.0.0&c=WEB_EMBEDDED_PLAYER&cver=1.20240915.00.00&cplayer=UNIPLAYER&cos=Macintosh&cosver=10_15_7&cplatform=DESKTOP&hl=en_US&cr=US&len=1489.281&rtn=226&feature=oembed&afmt=251&idpj=0&ldpj=-38&rti=186&size=786%3A443%3A2&inview=1&st=32.153&et=72.154&muted=0&docid=QqTSnI0bfFk&ei=uNbqZrbbIc210_wPwaGZ8AM&plid=AAYiZNrh4T60uJGt&referrer=https%3A%2F%2Fwww.youtube.com%2Fembed%2FQqTSnI0bfFk%3Ffeature%3Doembed%26rel%3D0%26enablejsapi%3D1&of=nJiyl6tUJbE7C7O6bu7jVg
        """
        qp = self.query_params
        if qp.get('state') in ('playing', 'paused'):
            return self.media_id()
        return ''


__all__ = (
    'Ustream',
    'Vimeo',
    'YouTube',
)
