from django.contrib.auth import get_user_model
from django.test import TestCase

from ..api.serializers import JWKSEndpointTrustSerializer, PublicKeySerializer
from ..choices import STATUS
from .data import (
    PEM_PUBLIC_RSA,
    PEM_PUBLIC_ED25519,
    PEM_PUBLIC_RSA_INVALID,
    PEM_PUBLIC_ED25519_INVALID,
)
from ..models import JWKSEndpointTrust, PublicKey
User = get_user_model()


class TestJWKSEndpointTrustSerializer(TestCase):

    def setUp(self) -> None:
        self.user = User.objects.create_user(username="foo", is_superuser=True)
        self.user.refresh_from_db()

    def test_valid(self):
        serializer = JWKSEndpointTrustSerializer(data={
            'user': self.user.cid,
            'jwks_url': 'https://aspen.com/.well-known/jwks.json',
        })
        self.assertTrue(serializer.is_valid(), serializer.errors)
        cert = serializer.save()
        self.assertIsInstance(cert, JWKSEndpointTrust)

    def test_invalid(self):
        serializer = JWKSEndpointTrustSerializer(data={
            'user': self.user.cid,
            'jwks_url': 'https://aspen',
        })
        self.assertFalse(serializer.is_valid())
        self.assertIn('jwks_url', serializer.errors)


class TestPublicKeySerializer(TestCase):

    def setUp(self) -> None:
        self.user = User.objects.create_user(username="foo", is_superuser=True)
        self.user.refresh_from_db()

    def test_rsa_serializer(self):
        serializer = PublicKeySerializer(data={
            'name': 'RSA Cert',
            'user': self.user.cid,
            'key': PEM_PUBLIC_RSA.decode(),
            'status': STATUS.active,
        })
        self.assertTrue(serializer.is_valid(), serializer.errors)
        cert = serializer.save()
        self.assertIsInstance(cert, PublicKey)

    def test_ed25519_serializer(self):
        serializer = PublicKeySerializer(data={
            'name': 'ED25519 Cert',
            'user': self.user.cid,
            'key': PEM_PUBLIC_ED25519.decode(),
            'status': STATUS.active,
        })
        self.assertTrue(serializer.is_valid(), serializer.errors)
        cert = serializer.save()
        self.assertIsInstance(cert, PublicKey)

    def test_invalid_serializer(self):
        for invalid_cert in (
            'not-a-valid-cert',
            '-----BEGIN PUBLIC KEY-----\nstillnotvalid\n-----END PUBLIC KEY-----',
            PEM_PUBLIC_ED25519_INVALID,
            PEM_PUBLIC_RSA_INVALID,
            None,
            (),
            b'nonsense',
        ):
            serializer = PublicKeySerializer(data={
                'name': 'ED25519 Cert',
                'user': self.user.cid,
                'key': invalid_cert,
                'status': STATUS.active,
            })
            self.assertFalse(serializer.is_valid())
            self.assertIn('key', serializer.errors)
