import base64
from uuid import uuid1

from django.contrib.auth import get_user_model
from django.test import TestCase
from rest_framework import status
from rest_framework.test import APIClient

from ..models import PublicKey
from ..tokens import Token
from ..keys import RSAPrivateKey, Ed25519PrivateKey
from .. import get_setting

User = get_user_model()


class RestFrameworkSignedTokenAuthenticationTest(TestCase):
    def setUp(self):
        self.api_client = APIClient(enforce_csrf_checks=True)
        self.user = User.objects.create_user(username="foo", is_superuser=True)

        self.key_ed25519 = Ed25519PrivateKey.generate()
        self.key_rsa = RSAPrivateKey.generate()

        self.user_key_ed25519 = PublicKey.objects.create(
            name='ed255919-drf',
            user=self.user,
            key=self.key_ed25519.public_key.as_pem.decode(),
        )
        self.user_key_ed25519.refresh_from_db()  # needed to reload CID?!

        self.user_key_rsa = PublicKey.objects.create(
            name='rsa-drf',
            user=self.user,
            key=self.key_rsa.public_key.as_pem.decode(),
        )
        self.user_key_rsa.refresh_from_db()

    def test_no_auth_unauthenticated_user(self):
        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/{self.user_key_ed25519.cid}/',
        )
        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED)

    def test_invalid_auth_unauthenticated_user(self):
        auth = 'JWT what-a-crazy-token'
        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/{self.user_key_ed25519.cid}/',
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED)

    def test_unverifiable_key_unauthenticated_user(self):
        auth = 'JWT eyJ0eXAiOiJKV1QiLCJhbGciOiJFZERTQSIsImtpZCI6IjJiNDIyZWNkZjM3YTYxNjVkOGRiMTY0MzE2N2UxNjEyYmZlNjE0ZmJmYWQwYmRiY2E2NjJkMGIyMDVlN2UxNGIifQ.eyJ1c2VybmFtZSI6ImZvbyIsInRpbWUiOjE2NDczNjg2MzksIm5vbmNlIjoibmNYTlpTaHlrWUkifQ.hNUay432cUdfUDYAfcRAvIpnWHPpmJK2saLdLvQAyllTDnzL9dpzehjv6uULOabTnzL_9YzulOnS4jXPh-tpCg'
        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/{self.user_key_ed25519.cid}/',
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED)

    def test_signed_token_authenticates_user(self):
        auth = Token(username='foo').create_auth_header(self.key_rsa)
        self.assertTrue(self.user.has_perm('signed_jwt_auth.view_publickey'))

        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/{self.user_key_rsa.cid}/',
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK)

    def test_auth_requesting_correct_public_key_succeeds(self):
        auth = Token(
            username='foo',
            public_key_cid=str(self.user_key_ed25519.cid),
        ).create_auth_header(self.key_ed25519)

        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/{self.user_key_ed25519.cid}/',
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK, response.json())

    def test_auth_requesting_nonexistent_public_key_fails(self):
        auth = Token(username='foo', public_key_cid=str(uuid1())).create_auth_header(self.key_rsa)

        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/{self.user_key_rsa.cid}/',
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED, response.json())

    def test_auth_requesting_nonexistent_mismatch_public_key_fails(self):
        """
        When a PublicKey Canonical ID is provided that actually does belong to
        the user, but which is NOT used to sign the token, then auth should fail.
        """
        auth = Token(
            username='foo',
            public_key_cid=str(self.user_key_ed25519.cid),
        ).create_auth_header(self.key_rsa)

        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/{self.user_key_rsa.cid}/',
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED, response.json())


class SignedTokenByProvidedKeyTest(TestCase):
    def setUp(self):
        self.api_client = APIClient(enforce_csrf_checks=True)
        self.user = User.objects.create_user(username="needs-public-key", is_superuser=True)
        self.key_ed25519 = Ed25519PrivateKey.generate()

    def test_attached_public_key_authenticates_user(self):
        auth = Token(username='needs-public-key').create_auto_auth_header(self.key_ed25519)
        self.assertTrue(self.user.has_perm('signed_jwt_auth.view_publickey'))
        public_key = base64.urlsafe_b64encode(self.key_ed25519.public_key.as_pem).decode()

        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/?verify_public_key={public_key}',
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK)

    def test_posting_to_endpoint_with_valid_verification_key_succeeds(self):
        """
        Posting to API with token signed by value included in `verify_public_key`
        creates the PublicKey record.
        """
        user = User.objects.create_user(username="register-public-key", is_superuser=True)
        self.assertFalse(user.public_keys.exists(), msg='User should have NO public keys!')
        user.refresh_from_db()

        auth = Token(username='register-public-key').create_auto_auth_header(self.key_ed25519)
        param_key = get_setting('AUTO_AUTH_METHOD_PARAM')

        data = {
            'name': 'auto-register-key',
            'user': user.cid,
            'key': self.key_ed25519.public_key.as_pem.decode(),
            param_key: self.key_ed25519.public_key.as_pem.decode(),
        }
        response = self.api_client.post(
            f'/api/signed_jwt_auth/publickey/',
            data=data,
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_201_CREATED, response.json())
        self.assertTrue(user.public_keys.exists())

    def test_posting_to_endpoint_with_preexisting_endpoint_fails(self):
        """
        Posting to API with token signed by value included in `verify_public_key`
        doesn't create PublicKey if user already has a PublicKey.
        """
        user = User.objects.create_user(username="register-public-key", is_superuser=True)
        existing = Ed25519PrivateKey.generate()
        PublicKey.objects.create(
            name='existing-key',
            user=user,
            key=existing.public_key.as_pem.decode(),
        )

        auth = Token(username='register-public-key').create_auto_auth_header(self.key_ed25519)
        param_key = get_setting('AUTO_AUTH_METHOD_PARAM')

        data = {
            'name': 'auto-register-key',
            'user': user.cid,
            'key': self.key_ed25519.public_key.as_pem.decode(),
            param_key: self.key_ed25519.public_key.as_pem.decode(),
        }
        response = self.api_client.post(
            f'/api/signed_jwt_auth/publickey/',
            data=data,
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED, response.json())

    def test_posting_to_endpoint_with_invalid_verification_key_fails(self):
        """
        Posting to API with token that is NOT signed by value included in `verify_public_key`
        must be unauthorized and refuse to create PublicKey record.
        """
        user = User.objects.create_user(username="register-public-key", is_superuser=True)
        self.assertFalse(user.public_keys.exists(), msg='User should have NO public keys!')
        user.refresh_from_db()

        new_key = Ed25519PrivateKey.generate()
        auth = Token(username='register-public-key').create_auto_auth_header(new_key)
        param_key = get_setting('AUTO_AUTH_METHOD_PARAM')

        data = {
            'name': 'auto-register-key',
            'user': user.cid,
            'key': new_key.as_pem.decode(),
            param_key: self.key_ed25519.public_key.as_pem.decode(),
        }
        response = self.api_client.post(
            f'/api/signed_jwt_auth/publickey/',
            data=data,
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED, response.json())
        self.assertFalse(user.public_keys.exists())


class SignedTokenByProvidedKeyOtpTest(TestCase):

    def setUp(self):
        self.api_client = APIClient(enforce_csrf_checks=True)
        self.user = User.objects.create_user(username="provides-otp-pk", is_superuser=True)
        self.user.refresh_from_db()
        self.key_ed25519 = Ed25519PrivateKey.generate()

    def test_attached_public_key_authenticates_user_with_no_public_key(self):
        auth = Token(username='provides-otp-pk').create_auto_auth_header(self.key_ed25519, True)
        self.assertTrue(self.user.has_perm('signed_jwt_auth.view_publickey'))
        public_key = base64.urlsafe_b64encode(self.key_ed25519.public_key.as_pem).decode()

        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/?verify_public_key={public_key}',
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK, response.json())

    def test_attached_public_key_authenticates_user_with_existing_public_key(self):
        """
        When a user has a pre-existing public key, but makes a request that includes
        an OTP in the token, then it succeeds.
        """
        user = User.objects.create_user(username="has-pk-otp-pk", is_superuser=True)
        existing = Ed25519PrivateKey.generate()
        PublicKey.objects.create(
            name='existing-key',
            user=user,
            key=existing.public_key.as_pem.decode(),
        )

        auth = Token(username='has-pk-otp-pk').create_auto_auth_header(self.key_ed25519, True)
        self.assertTrue(self.user.has_perm('signed_jwt_auth.view_publickey'))
        public_key = base64.urlsafe_b64encode(self.key_ed25519.public_key.as_pem).decode()

        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/?verify_public_key={public_key}',
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_200_OK, response.json())

        # Removing the OTP token results in authentication failure
        new_ed25519 = Ed25519PrivateKey.generate()
        auth = Token(username='has-pk-otp-pk').create_auto_auth_header(new_ed25519)
        self.assertTrue(self.user.has_perm('signed_jwt_auth.view_publickey'))
        public_key = base64.urlsafe_b64encode(new_ed25519.public_key.as_pem).decode()

        response = self.api_client.get(
            f'/api/signed_jwt_auth/publickey/?verify_public_key={public_key}',
            HTTP_AUTHORIZATION=auth,
        )
        self.assertEqual(response.status_code, status.HTTP_401_UNAUTHORIZED, response.json())
