from typing import Union
from django.contrib.auth import get_user_model
from rest_framework import exceptions
from .. import models
from ..tokens import UntrustedToken, Token
from ..typehints import ClaimCreds

User = get_user_model()


class BaseUserRepository:
    def get_user(self, lookups: ClaimCreds) -> Union[None, User]:  # pragma: no cover
        raise NotImplementedError()


class BasePublicKeyRepository:

    def attempt_to_verify_token(
            self,
            user: User,
            untrusted_token: UntrustedToken,
    ) -> Union[Token, None]:  # pragma: no cover
        """
        When a token includes the Canonical ID of the Public Key,
        then verify against _only_ that PublicKey.
        """
        requested_public_key = untrusted_token.requested_public_key()
        if not requested_public_key:
            return None

        user_key = models.PublicKey.objects.scid(requested_public_key)
        if user_key:
            public_key = user_key.get_key()
            token = untrusted_token.verify(public_key=public_key)
            if token:
                user_key.update_last_used_datetime()
                return token
            raise exceptions.AuthenticationFailed('Token not signed by the specified public key')

        raise exceptions.AuthenticationFailed('Requested public key does not exist')
