from django.conf import settings
from django.db import models
from django.utils import timezone
from django.utils.translation import gettext_lazy as _
from django.utils.encoding import force_str, force_bytes
from jwt import PyJWKClient
from typing import Optional

from console_base.theme import ICONS
from console_base.managers import LCBaseQuerySet
from console_base.models import BaseUUIDPKModel, LCDateTimeField, LCTextField, NameCITextField
from console_base.models.fernet import EncryptedBinaryField

from .choices import STATUS
from .fields import KeyField
from .managers import PublicKeyQuerySet
from .utils import get_key_password
from .validators import validate_public_key
from . import keys, tokens


class PublicKey(BaseUUIDPKModel):
    """
    Store a public key and associate it to a particular user.

    Implements the same concept as the OpenSSH ``~/.ssh/authorized_keys`` file on a Unix system.
    """
    css_icon = ICONS.TLS
    STATUS = STATUS

    #: Foreign key to the Django User model. Related name: ``public_keys``.
    user = models.ForeignKey(
        settings.AUTH_USER_MODEL,
        verbose_name=_("User"),
        related_name="public_keys",
        on_delete=models.CASCADE,
    )

    #: Key text in either PEM or OpenSSH format.
    key = KeyField(
        _("Public Key"),
        help_text=_("The user's RSA/Ed25519 public key"),
        validators=[validate_public_key],
    )

    #: Private Key bytes in either PEM or OpenSSH format.
    # Just because this field is on the model doesn't mean we should use it.
    # Included here for fast backup / restores of cloud systems, which to accomplish
    # means we need to store the private key on disk anyway. So it's saved in the
    # database for convenience, but should be rotated regularly.
    private_key = EncryptedBinaryField(
        _("Private Key"),
        null=True,
        blank=True,
        help_text=_("The user's RSA/Ed25519 private key"),
    )

    #: Info describing the key. What system is authenticating with the key etc.
    name = NameCITextField(_("Name"))

    #: Date and time that key was last used for authenticating a request.
    last_used_on = LCDateTimeField(_("Last Used On"), null=True, blank=True)
    revocation_date = LCDateTimeField(_("Revocation Date"), null=True, blank=True)
    status = LCTextField(_("Status"), choices=STATUS.choices, default=STATUS.active)

    objects = PublicKeyQuerySet.as_manager()

    class Meta:
        verbose_name = _("Public Key")
        verbose_name_plural = _("Public Keys")
        constraints = [
            models.UniqueConstraint(
                fields=('user', 'name'),
                name='unique_name_per_user',
            ),
            models.UniqueConstraint(
                fields=('key',),
                name='keys_must_be_unique',
            ),
        ]

    def __str__(self):
        return f'{self.__class__.__name__}({self.name!r})'

    def get_key(self) -> keys.FacadePublicKey:
        key_bytes = force_bytes(self.key)
        exc, key = keys.PublicKey.load_serialized_public_key(key_bytes)
        if key is None:
            if exc is None:  # pragma: no cover
                raise ValueError("Failed to load key")
            raise exc
        return key

    def get_private_key(self) -> Optional[keys.FacadePrivateKey]:
        openssh_banner = b'-----BEGIN OPENSSH PRIVATE KEY-----'
        password = get_key_password(self.cid, self.private_key.startswith(openssh_banner))
        if not password:
            return
        return keys.PrivateKey.load_pem(self.private_key, password=password)

    def update_last_used_datetime(self) -> None:
        self.last_used_on = timezone.now()
        self.save(update_fields=["last_used_on"])

    def save(self, *args, **kwargs) -> None:
        if not self.name:
            key_parts = force_str(self.key).split(" ")
            if len(key_parts) == 3:
                self.name = key_parts.pop()
        super().save(*args, **kwargs)


class JWKSEndpointTrust(BaseUUIDPKModel):
    """
    Associate a JSON Web Key Set (JWKS) URL with a Django User.

    This accomplishes the same purpose of the PublicKey model, in a more automated
    fashion. Instead of manually assigning a public key to a user, the system will
    load a list of public keys from this URL.
    """
    css_icon = ICONS.USER

    #: Foreign key to the Django User model. Related name: ``public_keys``.
    user = models.OneToOneField(
        settings.AUTH_USER_MODEL,
        verbose_name=_("User"),
        related_name="jwks_endpoint",
        on_delete=models.CASCADE,
    )

    #: URL of the JSON Web Key Set (JWKS)
    jwks_url = models.URLField(
        _("JSON Web Key Set (JWKS) URL"),
        help_text=_("e.g. https://dev-87evx9ru.auth0.com/.well-known/jwks.json"),
    )

    objects = LCBaseQuerySet.as_manager()

    class Meta:
        verbose_name = _("JSON Web Key Set")
        verbose_name_plural = _("JSON Web Key Sets")
        constraints = [
            models.UniqueConstraint(
                fields=('user', 'jwks_url'),
                name='unique_endpoint_url_per_user',
            ),
        ]

    @property
    def jwks_client(self) -> PyJWKClient:
        return PyJWKClient(self.jwks_url)

    def get_signing_key(self, untrusted_token: tokens.UntrustedToken) -> keys.PublicKey:
        jwk = self.jwks_client.get_signing_key_from_jwt(untrusted_token.token)
        return keys.PublicKey.from_cryptography_pubkey(jwk.key)
