from sequential_uuids.generators import uuid_time_nextval
from django.contrib.auth import get_user_model
from .utils import get_key_password

from console_base.managers import LCBaseQuerySet

from . import keys
from .choices import STATUS

User = get_user_model()


class PublicKeyQuerySet(LCBaseQuerySet):

    def valid(self):
        return self.active().filter(status=STATUS.active)

    def create_rsa_keys(self, name: str, user: User):
        """
        Create a new RSA Private / Public Key Pair record for the specified user.
        """
        return self._create_key_pair(
            key_cls=keys.RSAPrivateKey,
            name=name,
            user=user,
        )

    def create_ssh_keys(self, name: str, user: User):
        """
        Create a new SSH Private / Public Key Pair record for the specified user.
        """
        return self._create_key_pair(
            key_cls=keys.Ed25519PrivateKey,
            name=name,
            user=user,
        )

    def _create_key_pair(self, key_cls, name: str, user: User):
        """
        Helper method; don't call directly.
        Create new SSH or RSA Public / Private key pair for specified user.
        """
        cid = uuid_time_nextval()
        private_key = key_cls.generate()
        password = get_key_password(cid, ssh=key_cls == keys.Ed25519PrivateKey)

        if not password:
            return

        private_bytes = private_key.private_bytes(password)

        return self.create(
            name=name,
            user=user,
            cid=cid,
            key=private_key.public_key.as_pem.decode(),
            private_key=private_bytes,
        )
