from django.utils.translation import gettext_lazy as _
from drf_spectacular.extensions import OpenApiAuthenticationExtension
from .. import get_setting


class SignedTokenAuthenticationScheme(OpenApiAuthenticationExtension):
    target_class = 'signed_jwt_auth.authentication.SignedTokenAuthentication'
    name = 'SignedJwtAuth'

    def get_security_definition(self, auto_schema):
        return {
            'type': 'apiKey',
            'in': 'header',
            'name': 'Authorization',
            'description': base_description % {
                'method': get_setting('AUTH_METHOD'),
                'further_intro': intro_signed_key,
                'further_description': '',
                'user_claims': user_claims,
                'verification': verification_keys_signed,
            },
        }


class SignedTokenByProvidedKeyScheme(OpenApiAuthenticationExtension):
    target_class = 'signed_jwt_auth.authentication.SignedTokenByProvidedKey'
    name = 'SignedProvidedKeyJwtAuth'

    def get_security_definition(self, auto_schema):
        return {
            'type': 'apiKey',
            'in': 'header',
            'name': 'Authorization',
            'description': base_description % {
                'method': get_setting('AUTO_AUTH_METHOD'),
                'further_intro': intro_provided_key,
                'further_description': further_description_provided_key,
                'user_claims': user_claims,
                'verification': verification_keys_provided_key,
            },
        }


base_description = _("""
Signed JWT in authorization header - `Authorization %(method)s <signed-token>`. 

**A new token must be created signed for every http request!**

API Client must sign JWTs with an RSA or Ed25519 key. The API Server must a have Public Key record
assigned to the authenticating user. %(further_intro)s

%(further_description)s

## Token Payload

#### Verification Data
%(verification)s

%(user_claims)s
""")


user_claims = _("""
Token data must include one or more of the following keys to lookup the user 
(in the order of preference)

#### User Claims (one or more required)
| Key        | Description                 |
|------------|-----------------------------|
| `cid`      | Canonical ID of User record |
| `email`    | User's email address        |
| `username` | User's username / nickname  | 
""")


verification_keys_signed = _("""
| Key      | Description                                    |
|----------|------------------------------------------------|
| `time`   | UTC timestamp of date when token was generated |
| `nonce`  | Cryptographically secure random secret to avoid replay attacks |
""")


verification_keys_provided_key = _("""
| Key      | Description                                  |
|----------|----------------------------------------------|
| `time`   | UTC timestamp of date when token was generated |
| `nonce`  | Cryptographically secure random secret to avoid replay attacks |
| `otp`    | Correctly formed OTP. _(Contact support for the way to generate a valid OTP)_. |
""")

intro_signed_key = _("""
_This auth method is deprecated. Please switch to Paseto Tokens as soon as possible._
""")

intro_provided_key = _("""
Creating a new Public Key for a user without requiring pre-existing login credentials is
at times convenient. This can be done by including the Public Key that signed the JWT as a 
URL query parameter.

_Use this method sparingly! IE registering new Public Keys in machine-to-machine applications._
""")

further_description_provided_key = _("""

#### Query Parameter
| Key  | Description                                  |
|------|----------------------------------------------|
| `%s` | Public Key (base64-encoded) for validating the JWT of request. |
""") % get_setting('AUTO_AUTH_METHOD_PARAM')


__all__ = (
    'SignedTokenAuthenticationScheme',
    'SignedTokenByProvidedKeyScheme',
)
