# Types for default, basic responses from Redwood API calls
from enum import StrEnum
from msgspec import Struct
from .categories import CATEGORY_NAMES_SCORES, RuleStat, RULE_NAMES_COUNT


class ApiFunction(StrEnum):
    """
    Classify API function / endpoint to call
    """

    URL = "classify"
    Text = "classify-text"
    Tally = "analyze-tally"


class ApiMode(StrEnum):
    Normal = ""
    Verbose = "verbose"
    Analyze = "analyze-score"


class ClassifyTally(Struct, kw_only=True):
    """
    Return type from Redwood's "/analyze-tally" API.
    """

    rules: RULE_NAMES_COUNT
    scoreAnalysis: dict[str, dict[str, RuleStat]]


class CategoriesType(Struct):
    """
    Base type with required category score dict.
    """

    categories: CATEGORY_NAMES_SCORES


class ClassifyText(CategoriesType):
    """
    Return type from Redwood's "/classify-text/verbose" API.
    """

    text: str
    rules: RULE_NAMES_COUNT


class ClassifyUrl(CategoriesType):
    """
    Return type from Redwood's "/classify/verbose" API.

    {
        'url': 'http://lxer.com',
        'categories': {'computer': 1380, 'news': 249, 'os': 830},
        'logLine': ['2023-10-28 15:46:16.128089', '', 'classify', 'http://lxer.com', 'GET', '200', 'text/html', '50580', '', '< linux > 74, <story type> 60, < tutorial> 19, < ubuntu> 19, < debian > 17, <news story> 14, < gnome > 9, <laptop> 8, <open source> 8, < nvidia > 4, < tuxedo> 4, <cinnamon> 4, <ip address> 4, < gaming> 3, < repository> 3, <mysql> 3, <software develop> 3, < games> 2, < ryzen > 2, <bittorrent> 2, <dual boot> 2, <latest news> 2, <shell script> 2, <visual studio> 2, <web develop> 2, /./ 1, < door > 1, < i2p > 1, < iso > 1, < kde > 1, < unix > 1, < witch> 1, <bug fix> 1, <data privacy> 1, <desktop environ> 1, <fruit> 1, <gamepad> 1, <industrial control> 1, <lenovo> 1, <linux kernel> 1, <machine learning> 1, <major release> 1, <name search> 1, <raspberry pi> 1, <raspberry> 1, <tesla> 1, <user interface> 1, <virtualization> 1', 'computer 1380, os 830, news 249', '', '', '', 'Mozilla/5.0 (Macintosh; Intel Mac OS X 11_7_10) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/119.0.6045.53 Safari/537.36', 'HTTP/1.1', '', 'Macintosh', '', '', '', ''],
    }
    """

    url: str
    logLine: list[str]


class ClassifyUrlAnalyze(CategoriesType):
    """
    Return type from Redwood's "/classify/analyze-score" API.

    {
        'url': 'http://lxer.com',
        'categories': {'computer': 1380, 'news': 249, 'os': 830},
        'scoreAnalysis': {
            'ai': {'<machine learning>': {'Count': 1, 'Score': 30}},
            'audioelectronics': {'<industrial control>': {'Count': 1, 'Score': 30}},
            'auto': {'<tesla>': {'Count': 1, 'Score': 25}},
            'clothing': {'< tuxedo>': {'Count': 4, 'Score': 120}},
            'computer': {'< games>': {'Count': 2, 'Score': -20}, '< i2p >': {'Count': 1, 'Score': 195}, '< nvidia >': {'Count': 4, 'Score': 40}, '< repository>': {'Count': 3, 'Score': 60}, '< ubuntu>': {'Count': 19, 'Score': 250}, '<bug fix>': {'Count': 1, 'Score': 15}, '<data privacy>': {'Count': 1, 'Score': 15}, '<desktop environ>': {'Count': 1, 'Score': 30}, '<dual boot>': {'Count': 2, 'Score': 54}, '<ip address>': {'Count': 4, 'Score': 60}, '<laptop>': {'Count': 8, 'Score': 160}, '<lenovo>': {'Count': 1, 'Score': 15}, '<linux kernel>': {'Count': 1, 'Score': 20}, '<machine learning>': {'Count': 1, 'Score': 25}, '<major release>': {'Count': 1, 'Score': 10}, '<mysql>': {'Count': 3, 'Score': 90}, '<open source>': {'Count': 8, 'Score': 120}, '<raspberry pi>': {'Count': 1, 'Score': 25}, '<shell script>': {'Count': 2, 'Score': 54}, '<software develop>': {'Count': 3, 'Score': 75}, '<user interface>': {'Count': 1, 'Score': 10}, '<virtualization>': {'Count': 1, 'Score': 27}, '<visual studio>': {'Count': 2, 'Score': 50}},
            'construction': {'< door >': {'Count': 1, 'Score': 10}},
            'energyelectrical': {'<industrial control>': {'Count': 1, 'Score': 30}},
            'foods': {'<fruit>': {'Count': 1, 'Score': 5}}, 'foods/beverages': {'<fruit>': {'Count': 1, 'Score': 5}},
            'games': {'< games>': {'Count': 2, 'Score': 30}, '< gaming>': {'Count': 3, 'Score': 45}, '<gamepad>': {'Count': 1, 'Score': 45}},
            'herbal': {'<cinnamon>': {'Count': 4, 'Score': 40}, '<raspberry>': {'Count': 1, 'Score': 10}}, 'lc_Cortana_block': {'/./': {'Count': 1, 'Score': 1500}}, 'lc_matchallNone': {'/./': {'Count': 1, 'Score': 1500}}, 'news': {'<latest news>': {'Count': 2, 'Score': 50}, '<news story>': {'Count': 14, 'Score': 199}},
            'os': {'< debian >': {'Count': 17, 'Score': 340}, '< gnome >': {'Count': 9, 'Score': 243}, '< kde >': {'Count': 1, 'Score': 30}, '< linux >': {'Count': 74, 'Score': 190}, '< unix >': {'Count': 1, 'Score': 27}},
            'training': {'< tutorial>': {'Count': 19, 'Score': 95}}, 'training/safety': {'< tutorial>': {'Count': 19, 'Score': 95}},
        },
    }
    """

    url: str
    scoreAnalysis: dict[str, dict[str, RuleStat]]


class ApiError(Struct):
    status_code: int
    message: str


__all__ = (
    "CategoriesType",
    "ClassifyTally",
    "ClassifyText",
    "ClassifyUrl",
    "ClassifyUrlAnalyze",
    "RuleStat",
    "ApiFunction",
    "ApiMode",
)
