from msgspec import Struct
from django.db.models.enums import TextChoices
from .categories import NUMBER


class AutofixTier(TextChoices):
    Level1 = "level1", "Level 1"
    Level2 = "level2", "Level 2"
    Level3 = "level3", "Level 3"
    Level4 = "level4", "Level 4"


class AutofixMode(TextChoices):
    Allow = "allow", "Allow Entire Category"
    Offset = "offset", "Allow only the blocked website"
    Block = "block", "Block Autofix Request"


class RatingRatio(Struct):
    """
    Combined ratio of URL / Phrase Scoring compared
    to total Score of the entire request.
    """

    Level1: NUMBER = 0.0
    Level2: NUMBER = 0.0
    Level3: NUMBER = 0.0
    Level4: NUMBER = 0.0
    BOULDER: NUMBER = 0.0  # only to track the presence of boulder categories!

    def confidence(self):
        """
        Return the highest confidence value from
        the combined ratings of all categories.
        """
        return max((self.Level1, self.Level2, self.Level3, self.Level4))


class AutofixMessage(TextChoices):
    """
    Message to return to the user to explain the Autofix Level calculation.
    """

    Unknown = "unknown", "Unable to calculate precise Autofix Level"

    # Level 1 Messages
    WellRated = "wellrated", "The category rates highly but was set to block"
    Satisfactory = (
        "satisfactory",
        "Acceptable categories were found in high ratio, and no blocking categories from worst ratings were found",
    )
    LowCount = (
        "lowcount",
        "Total page score is low and calculated from lower rule count",
    )
    Skewed = (
        "skewed",
        "Phrase scoring was skewed by a few, very negative phrases occurring unusually frequently",
    )

    # Level 2 Messages
    HighScoreRatio = (
        "highscoreratio",
        "Overall total score of well-rated categories compares favorably with total objectionable categories",
    )
    ModerateScoreRatio = (
        "moderatescoreratio",
        "Moderate categories favorably outscore objectionable categories",
    )
    PermitLevel2 = "permit2", "Category is specifically permitted at Level 2"

    # Level 3 Messages
    Questionable = (
        "questionable",
        "No acceptable categories found, but none of the most objectionable categories are found",
    )
    Doubtful = (
        "doubtful",
        "No acceptable categories found, but the phrase scoring could be more accurate",
    )
    RequireLevel3 = (
        "require3",
        "Category assigned to Level 3 regardless of secondary page content",
    )

    # Level 4 Messages
    Rough = "rough", "Secondary ratings are more objectionable than primary category"
    Worst = "worst", "Only the worst ratings of categories were found"
    RequireLevel4 = "require4", "Autofix is disabled for this category"

    # Correctly Scored Messages
    Confident = "confident", "The classification results are almost certainly correct"

    # Preference Messages
    NoPreference = "no_pref", "No Company or Policy preference defined"
    PolicyAllow = "policy_allow", "Allowed by policy preference"
    PolicyBlock = (
        "policy_block",
        "Your accountability policy disabled autofixing this category",
    )

    CompanyAllow = "company_allow", "Allowed by company preference"
    CompanyBlock = "company_block", "Your company disabled autofixing this category"

    Disabled = (
        "disabled",
        "Company or Policy preference disabled autofixing this category",
    )


__all__ = (
    "AutofixTier",
    "AutofixMode",
    "RatingRatio",
    "AutofixMessage",
)
