from unittest import TestCase
from redwoodctl.typehints import (
    AutofixMessage,
    RatingStat,
    ClassifierCategoryStat,
    RatingName,
    RedwoodAction,
)
from redwoodctl.redwood.analysis import (
    Answer,
    AutofixAnalysis,
    CategoryAnalysis,
    CombinedRatingAnalysis,
)
from .fixtures.analysis import (
    bargain_hunter_categories,
    bargain_hunter_ratings,
    monmouth_park_categories,
    monmouth_park_ratings,
    sports_betting_categories,
    sports_betting_ratings,
    nfl_categories,
    nfl_ratings,
    getty_boulder_categories,
    getty_boulder_ratings,
)
from ..settings import (
    LOW_CONFIDENCE_SCORE_THRESHOLD,
    MED_CONFIDENCE_SCORE_THRESHOLD,
    HIGH_CONFIDENCE_SCORE_THRESHOLD,
)


class TestCategoryAnalysis(TestCase):
    def test_level_1(self):
        rating_with_phrase_score = {
            "auto": ClassifierCategoryStat(
                score=837,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=450,
                phrase_rules=[],
                regex_score=0,
                regex_rules=[],
            ),
            "ag": ClassifierCategoryStat(
                score=450,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=450,
                domain_rules=["deere.com"],
                ip_score=0,
                ip_rules=[],
                phrase_score=0,
                phrase_rules=[],
                regex_score=0,
                regex_rules=[],
            ),
        }
        analysis = CategoryAnalysis(rating_with_phrase_score)
        self.assertEqual(analysis.top(), "auto")

    def test_level_2(self):
        rating_with_phrase_score = {
            "hunting": ClassifierCategoryStat(
                score=837,
                rating=RatingName.PEBBLE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=450,
                phrase_rules=[],
                regex_score=0,
                regex_rules=[],
            ),
        }
        analysis = CategoryAnalysis(rating_with_phrase_score)
        self.assertTrue(analysis.level_2_rated())

    def test_level_3(self):
        """
        When the content is Level4, but scoring skewed,
        then use Level3. Otherwise, it's Level4
        """
        score = MED_CONFIDENCE_SCORE_THRESHOLD - 5
        phishing = ClassifierCategoryStat(
            score=score,
            rating=RatingName.BOULDER,
            action=RedwoodAction.Block,
            domain_score=0,
            domain_rules=[],
            ip_score=0,
            ip_rules=[],
            phrase_score=score,
            phrase_rules=list(range(5)),
            regex_score=0,
            regex_rules=[],
        )
        self.assertTrue(phishing.phrase_scoring_skewed())
        categories = {"phishing": phishing}
        analysis = CategoryAnalysis(categories)
        self.assertTrue(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

        score = MED_CONFIDENCE_SCORE_THRESHOLD + 5
        not_skewed_celebrity = ClassifierCategoryStat(
            score=score,
            rating=RatingName.BOULDER,
            action=RedwoodAction.Block,
            domain_score=0,
            domain_rules=[],
            ip_score=0,
            ip_rules=[],
            phrase_score=score,
            phrase_rules=list(range(15)),
            regex_score=0,
            regex_rules=[],
        )
        self.assertFalse(not_skewed_celebrity.phrase_scoring_skewed())
        categories = {"celebrity": not_skewed_celebrity}
        analysis = CategoryAnalysis(categories)

        self.assertFalse(analysis.level_3_rated())
        self.assertTrue(analysis.level_4_rated())

    def test_level_4(self):
        phrase_score = MED_CONFIDENCE_SCORE_THRESHOLD + 5
        domain_score = 350
        score = phrase_score + domain_score
        celebrity = ClassifierCategoryStat(
            score=score,
            rating=RatingName.BOULDER,
            action=RedwoodAction.Block,
            domain_score=domain_score,
            domain_rules=[],
            ip_score=0,
            ip_rules=[],
            phrase_score=phrase_score,
            phrase_rules=list(range(15)),
            regex_score=0,
            regex_rules=[],
        )
        categories = {"celebrity": celebrity}

        self.assertFalse(celebrity.phrase_scoring_skewed())

        analysis = CategoryAnalysis(categories)
        self.assertFalse(analysis.level_3_rated())
        self.assertTrue(analysis.level_4_rated())

    def test_monmouth_park_com_categories(self):
        analysis = CategoryAnalysis(monmouth_park_categories)
        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertTrue(analysis.level_4_rated())


class TestCombinedRatingAnalysisLevel1(TestCase):
    def test_all_ratings_within_group(self):
        rating_with_phrase_score = {
            RatingName.SILT: RatingStat(total_score=932, phrase_score=432, phrase_count=17),
            RatingName.BASE: RatingStat(total_score=327, phrase_score=327, phrase_count=4),
        }
        analysis = CombinedRatingAnalysis(rating_with_phrase_score)
        self.assertEqual(analysis.top(), RatingName.SILT)

        self.assertTrue(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

        rating_without_phrase_score = {
            RatingName.SILT: RatingStat(total_score=932, phrase_score=0, phrase_count=0),
            RatingName.BASE: RatingStat(total_score=327, phrase_score=0, phrase_count=0),
        }
        analysis = CombinedRatingAnalysis(rating_without_phrase_score)
        self.assertTrue(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

    def test_includes_transition_rating_1(self):
        rating_with_transition_rating = {
            RatingName.SILT: RatingStat(total_score=1232, phrase_score=748, phrase_count=19),
            RatingName.SAND: RatingStat(total_score=1232, phrase_score=932, phrase_count=27),
        }
        analysis = CombinedRatingAnalysis(rating_with_transition_rating)
        self.assertTrue(analysis.level_1_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

    def test_only_transition_rating(self):
        rating_with_transition_rating = {
            RatingName.SAND: RatingStat(total_score=1232, phrase_score=756, phrase_count=14),
        }
        analysis = CombinedRatingAnalysis(rating_with_transition_rating)
        self.assertFalse(analysis.level_1_rated())
        self.assertTrue(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())


class TestCombinedRatingAnalysisLevel2(TestCase):
    def test_all_ratings_within_group(self):
        rating_with_phrase_score = {
            RatingName.SAND: RatingStat(total_score=932, phrase_score=432, phrase_count=13),
            RatingName.PEBBLE: RatingStat(total_score=327, phrase_score=327, phrase_count=12),
        }
        analysis = CombinedRatingAnalysis(rating_with_phrase_score)
        self.assertEqual(analysis.top(), RatingName.SAND)

        self.assertFalse(analysis.level_1_rated())
        self.assertTrue(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

        rating_without_phrase_score = {
            RatingName.SAND: RatingStat(total_score=932, phrase_score=0, phrase_count=0),
            RatingName.PEBBLE: RatingStat(total_score=327, phrase_score=0, phrase_count=0),
        }
        analysis = CombinedRatingAnalysis(rating_without_phrase_score)
        self.assertFalse(analysis.level_1_rated())
        self.assertTrue(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

    def test_only_transition_rating_without_phrase_score(self):
        # Level1-2 transition ratings without phrase score should be Level1.
        rating_without_phrase_score = {
            RatingName.SAND: RatingStat(total_score=932, phrase_score=0, phrase_count=0),
        }
        analysis = CombinedRatingAnalysis(rating_without_phrase_score)
        self.assertTrue(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

    def test_only_transition_rating_with_phrase_score(self):
        # Level1-2 transition ratings with phrase score should be Level2.
        rating_without_phrase_score = {
            RatingName.SAND: RatingStat(total_score=932, phrase_score=357, phrase_count=8),
        }
        analysis = CombinedRatingAnalysis(rating_without_phrase_score)
        self.assertFalse(analysis.level_1_rated())
        self.assertTrue(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

    def test_includes_transition_rating_2(self):
        """
        If well-rated content is almost equal to Stone content,
        and no Boulder content is found, then it's Level2.
        """
        rating_with_transition_rating = {
            RatingName.PEBBLE: RatingStat(total_score=1232, phrase_score=748, phrase_count=19),
            RatingName.STONE: RatingStat(total_score=1275, phrase_score=932, phrase_count=23),
        }
        analysis = CombinedRatingAnalysis(rating_with_transition_rating)
        self.assertFalse(analysis.level_1_rated())
        self.assertTrue(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())


class TestLevelTwoLevelThreeScoreConfidenceTransition(TestCase):
    """
    The STONE rating can flip between Level2 and Level3.

    If the phrase scoring is skewed, consider it Level2.

    If we're confident in phrase scoring, it's Level3.
    """

    def test_only_transition_rating_high_confidence(self):
        """
        When a transition category has high confidence
        in the Scoring, call it Level3.
        """
        rating_good_confidence = {
            RatingName.STONE: RatingStat(total_score=1232, phrase_score=756, phrase_count=11),
        }
        self.assertFalse(rating_good_confidence[RatingName.STONE].phrase_scoring_skewed())

        analysis = CombinedRatingAnalysis(rating_good_confidence)
        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertTrue(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

    def test_only_transition_rating_low_confidence(self):
        """
        When a transition category has low confidence
        in the Scoring, call it Level2.
        """
        rating_low_confidence = {
            RatingName.STONE: RatingStat(total_score=1232, phrase_score=756, phrase_count=3),
        }

        self.assertTrue(rating_low_confidence[RatingName.STONE].phrase_scoring_skewed())
        analysis = CombinedRatingAnalysis(rating_low_confidence)
        self.assertFalse(analysis.level_1_rated())
        self.assertTrue(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

    def test_only_transition_rating_low_confidence_boulder(self):
        """
        When a transition category has low confidence in the Scoring,
        and also a Level4 rating, call it Level3 in spite of the skew.
        """
        rating_low_confidence = {
            RatingName.STONE: RatingStat(total_score=1232, phrase_score=756, phrase_count=3),
            RatingName.SAND: RatingStat(total_score=1097, phrase_score=1097, phrase_count=31),
            RatingName.ROCK: RatingStat(total_score=627, phrase_score=627, phrase_count=21),
        }

        self.assertTrue(rating_low_confidence[RatingName.STONE].phrase_scoring_skewed())
        analysis = CombinedRatingAnalysis(rating_low_confidence)
        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertTrue(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())


class TestCombinedRatingAnalysisLevel3(TestCase):
    """
    Level3 is a very small slice between Level2 and Level4.
    The only Level3 rating is "stone", and Level3 applies
    only if sole rating is "stone", and has phrase scoring.
    """

    def test_moderately_good_and_very_bad(self):
        rating_with_phrase_score = {
            RatingName.PEBBLE: RatingStat(total_score=932, phrase_score=682, phrase_count=13),
            RatingName.BOULDER: RatingStat(total_score=289, phrase_score=289, phrase_count=3),
        }
        analysis = CombinedRatingAnalysis(rating_with_phrase_score)
        self.assertEqual(analysis.top(), RatingName.PEBBLE)

        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertTrue(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

    def test_ratings_including_four(self):
        rating_with_phrase_score = {
            RatingName.PEBBLE: RatingStat(total_score=932, phrase_score=682, phrase_count=13),
            RatingName.STONE: RatingStat(total_score=932, phrase_score=682, phrase_count=13),
            RatingName.BOULDER: RatingStat(total_score=389, phrase_score=389, phrase_count=6),
        }
        analysis = CombinedRatingAnalysis(rating_with_phrase_score)
        self.assertEqual(analysis.top(), RatingName.PEBBLE)

        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertTrue(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

    def test_rating_spread(self):
        """
        When a boulder category contains well-rated categories, consider it Level3
        """
        ratings = {
            RatingName.BOULDER: RatingStat(total_score=875, phrase_score=375, phrase_count=12),
            RatingName.SILT: RatingStat(total_score=355, phrase_score=355, phrase_count=17),
            RatingName.BASE: RatingStat(total_score=365, phrase_score=365, phrase_count=13),
            RatingName.SAND: RatingStat(total_score=297, phrase_score=297, phrase_count=9),
        }
        analysis = CombinedRatingAnalysis(ratings)
        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertTrue(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())

    def test_monmouth_park_com_ratings(self):
        analysis = CombinedRatingAnalysis(monmouth_park_ratings)
        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertTrue(analysis.level_3_rated())
        self.assertFalse(analysis.level_4_rated())


class TestCombinedRatingAnalysisLevel4(TestCase):
    """
    Level 4 represents the coarsest ratings,
    and should never be autofix-able.
    """

    def test_all_ratings_within_group(self):
        rating_with_phrase_score = {
            RatingName.ROCK: RatingStat(total_score=932, phrase_score=432, phrase_count=5),
            RatingName.STONE: RatingStat(total_score=327, phrase_score=327, phrase_count=4),
        }
        analysis = CombinedRatingAnalysis(rating_with_phrase_score)
        self.assertEqual(analysis.top(), RatingName.ROCK)

        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertTrue(analysis.level_4_rated())

        rating_without_phrase_score = {
            RatingName.ROCK: RatingStat(total_score=932, phrase_score=0, phrase_count=0),
            RatingName.STONE: RatingStat(total_score=327, phrase_score=0, phrase_count=0),
        }
        analysis = CombinedRatingAnalysis(rating_without_phrase_score)
        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertTrue(analysis.level_4_rated())

    def test_includes_transition_rating_4(self):
        rating_with_transition_rating = {
            RatingName.BOULDER: RatingStat(total_score=1232, phrase_score=748, phrase_count=28),
            RatingName.STONE: RatingStat(total_score=1232, phrase_score=932, phrase_count=11),
        }
        analysis = CombinedRatingAnalysis(rating_with_transition_rating)
        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertTrue(analysis.level_4_rated())

    def test_includes_all_ratings_but_majority_rough(self):
        ratings = {
            RatingName.BOULDER: RatingStat(total_score=2139, phrase_score=2139, phrase_count=32),
            RatingName.ROCK: RatingStat(total_score=928, phrase_score=928, phrase_count=29),
            RatingName.PEBBLE: RatingStat(total_score=935, phrase_score=935, phrase_count=25),
            RatingName.STONE: RatingStat(total_score=820, phrase_score=820, phrase_count=23),
            RatingName.SILT: RatingStat(total_score=780, phrase_score=780, phrase_count=37),
            RatingName.SAND: RatingStat(total_score=520, phrase_score=520, phrase_count=21),
        }
        analysis = CombinedRatingAnalysis(ratings)
        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertTrue(analysis.level_4_rated())

    def test_includes_all_ratings_but_super_majority_rough(self):
        ratings = {
            RatingName.BOULDER: RatingStat(total_score=75939, phrase_score=75939, phrase_count=157),
            RatingName.ROCK: RatingStat(total_score=3028, phrase_score=3028, phrase_count=87),
            RatingName.PEBBLE: RatingStat(total_score=835, phrase_score=835, phrase_count=28),
            RatingName.STONE: RatingStat(total_score=630, phrase_score=630, phrase_count=31),
            RatingName.SILT: RatingStat(total_score=480, phrase_score=480, phrase_count=18),
            RatingName.SAND: RatingStat(total_score=220, phrase_score=220, phrase_count=5),
        }
        analysis = CombinedRatingAnalysis(ratings)
        self.assertFalse(analysis.level_1_rated())
        self.assertFalse(analysis.level_2_rated())
        self.assertFalse(analysis.level_3_rated())
        self.assertTrue(analysis.level_4_rated())


def combine_ratings(categories):
    ratings = {}
    for cat, stats in categories.items():
        try:
            ratings[stats.rating].total_score += stats.score
            ratings[stats.rating].phrase_score += stats.phrase_score
            ratings[stats.rating].phrase_count += len(stats.phrase_rules)
        except KeyError:
            ratings[stats.rating] = RatingStat(
                total_score=stats.score,
                phrase_score=stats.phrase_score,
                phrase_count=len(stats.phrase_rules),
            )
    return ratings


class TestAutofixAnalysisLevel1(TestCase):
    def test_low_count(self):
        category_score = MED_CONFIDENCE_SCORE_THRESHOLD - 5
        ag_score = LOW_CONFIDENCE_SCORE_THRESHOLD
        categories = {
            "auto": ClassifierCategoryStat(
                score=category_score,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=category_score,
                phrase_rules=list(range(5)),
                regex_score=0,
                regex_rules=[],
            ),
            "ag": ClassifierCategoryStat(
                score=ag_score,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=ag_score,
                phrase_rules=list(range(3)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["auto"])
        self.assertTrue(autofix_analysis.top_category_stats.phrase_scoring_skewed())
        self.assertEqual(autofix_analysis.is_level_1(), Answer(True, AutofixMessage.LowCount))

    def test_phrases_skewed(self):
        ag_score = MED_CONFIDENCE_SCORE_THRESHOLD + 150
        category_score = LOW_CONFIDENCE_SCORE_THRESHOLD
        categories = {
            "auto": ClassifierCategoryStat(
                score=category_score,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=category_score,
                phrase_rules=list(range(5)),
                regex_score=0,
                regex_rules=[],
            ),
            "ag": ClassifierCategoryStat(
                score=ag_score,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=ag_score,
                phrase_rules=list(range(3)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["ag"])
        self.assertTrue(autofix_analysis.top_category_stats.phrase_scoring_skewed())
        self.assertEqual(autofix_analysis.is_level_1(), Answer(True, AutofixMessage.Skewed))

    def test_well_rated(self):
        categories = {
            "auto": ClassifierCategoryStat(
                score=837,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=450,
                phrase_rules=[],
                regex_score=0,
                regex_rules=[],
            ),
            "ag": ClassifierCategoryStat(
                score=450,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=450,
                domain_rules=["deere.com"],
                ip_score=0,
                ip_rules=[],
                phrase_score=0,
                phrase_rules=[],
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["auto"])

        self.assertEqual(autofix_analysis.is_level_1(), Answer(True, AutofixMessage.WellRated))

    def test_satisfactory(self):
        categories = {
            "bikes": ClassifierCategoryStat(
                score=1837,
                rating=RatingName.SAND,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=1837,
                phrase_rules=list(range(40)),
                regex_score=0,
                regex_rules=[],
            ),
            "auto": ClassifierCategoryStat(
                score=837,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=450,
                phrase_rules=list(range(10)),
                regex_score=0,
                regex_rules=[],
            ),
            "ag": ClassifierCategoryStat(
                score=450,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=0,
                phrase_rules=[],
                regex_score=450,
                regex_rules=["/ag-bikes/d"],
            ),
            "construction": ClassifierCategoryStat(
                score=450,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=450,
                domain_rules=["homedepot.com"],
                ip_score=0,
                ip_rules=[],
                phrase_score=0,
                phrase_rules=[],
                regex_score=450,
                regex_rules=["/bike-roofer/d"],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["bikes"])
        self.assertEqual(autofix_analysis.is_level_1(), Answer(True, AutofixMessage.Satisfactory))


class TestAutofixAnalysisLevel2(TestCase):
    def test_video(self):
        category_score = MED_CONFIDENCE_SCORE_THRESHOLD - 5
        ag_score = LOW_CONFIDENCE_SCORE_THRESHOLD
        categories = {
            "content_youtube": ClassifierCategoryStat(
                score=category_score,
                rating=RatingName.STONE,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=category_score,
                phrase_rules=list(range(5)),
                regex_score=0,
                regex_rules=[],
            ),
            "ag": ClassifierCategoryStat(
                score=ag_score,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=ag_score,
                phrase_rules=list(range(3)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["content_youtube"])
        self.assertEqual(autofix_analysis.is_level_2(), Answer(True, AutofixMessage.PermitLevel2))

    def test_moderate_score_ratio(self):
        fitness_score = MED_CONFIDENCE_SCORE_THRESHOLD - 5
        bikes_score = LOW_CONFIDENCE_SCORE_THRESHOLD
        categories = {
            "powersports": ClassifierCategoryStat(
                score=fitness_score * 2,
                rating=RatingName.PEBBLE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=fitness_score * 2,
                phrase_rules=list(range(25)),
                regex_score=0,
                regex_rules=[],
            ),
            "fitness": ClassifierCategoryStat(
                score=fitness_score,
                rating=RatingName.PEBBLE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=fitness_score,
                phrase_rules=list(range(25)),
                regex_score=0,
                regex_rules=[],
            ),
            "bikes": ClassifierCategoryStat(
                score=bikes_score,
                rating=RatingName.PEBBLE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=bikes_score,
                phrase_rules=list(range(17)),
                regex_score=0,
                regex_rules=[],
            ),
            "sportsrules": ClassifierCategoryStat(
                score=bikes_score,
                rating=RatingName.PEBBLE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=bikes_score,
                phrase_rules=list(range(17)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["powersports"])
        self.assertEqual(
            autofix_analysis.is_level_2(),
            Answer(True, AutofixMessage.ModerateScoreRatio),
        )

    def test_high_score_ratio(self):
        construction_score = MED_CONFIDENCE_SCORE_THRESHOLD + 50
        fitness_score = MED_CONFIDENCE_SCORE_THRESHOLD - 5
        bikes_score = LOW_CONFIDENCE_SCORE_THRESHOLD
        categories = {
            "construction": ClassifierCategoryStat(
                score=construction_score,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=construction_score,
                phrase_rules=list(range(25)),
                regex_score=0,
                regex_rules=[],
            ),
            "fitness": ClassifierCategoryStat(
                score=fitness_score,
                rating=RatingName.PEBBLE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=fitness_score,
                phrase_rules=list(range(25)),
                regex_score=0,
                regex_rules=[],
            ),
            "bikes": ClassifierCategoryStat(
                score=bikes_score,
                rating=RatingName.PEBBLE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=bikes_score,
                phrase_rules=list(range(17)),
                regex_score=0,
                regex_rules=[],
            ),
            "sportsrules": ClassifierCategoryStat(
                score=bikes_score,
                rating=RatingName.PEBBLE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=bikes_score,
                phrase_rules=list(range(17)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["construction"])
        self.assertEqual(
            autofix_analysis.is_level_2(),
            Answer(True, AutofixMessage.HighScoreRatio),
        )


class TestAutofixAnalysisLevel3(TestCase):
    def test_confident_in_classifying(self):
        """
        When we're confident with the classifier results,
        Stone is Level3.
        """
        racing_score = MED_CONFIDENCE_SCORE_THRESHOLD - 5
        fashion_score = LOW_CONFIDENCE_SCORE_THRESHOLD + 10
        categories = {
            "racing": ClassifierCategoryStat(
                score=racing_score,
                rating=RatingName.STONE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=racing_score,
                phrase_rules=list(range(21)),
                regex_score=0,
                regex_rules=[],
            ),
            "fashion": ClassifierCategoryStat(
                score=fashion_score,
                rating=RatingName.BOULDER,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=fashion_score,
                phrase_rules=list(range(13)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["racing"])
        self.assertEqual(autofix_analysis.is_level_3(), Answer(True, AutofixMessage.Confident))

    def test_questionable(self):
        racing_score = MED_CONFIDENCE_SCORE_THRESHOLD - 5
        fashion_score = LOW_CONFIDENCE_SCORE_THRESHOLD + 10
        categories = {
            "racing": ClassifierCategoryStat(
                score=racing_score,
                rating=RatingName.STONE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=racing_score,
                phrase_rules=list(range(3)),
                regex_score=0,
                regex_rules=[],
            ),
            "fashion": ClassifierCategoryStat(
                score=fashion_score,
                rating=RatingName.BOULDER,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=fashion_score,
                phrase_rules=list(range(3)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["racing"])
        self.assertEqual(autofix_analysis.is_level_3(), Answer(True, AutofixMessage.Questionable))

    def test_doubtful_when_phrase_score_low(self):
        racing_score = LOW_CONFIDENCE_SCORE_THRESHOLD - 5
        fashion_score = MED_CONFIDENCE_SCORE_THRESHOLD - 10
        categories = {
            "fashion": ClassifierCategoryStat(
                score=fashion_score,
                rating=RatingName.BOULDER,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=fashion_score,
                phrase_rules=list(range(5)),
                regex_score=0,
                regex_rules=[],
            ),
            "racing": ClassifierCategoryStat(
                score=racing_score,
                rating=RatingName.STONE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=racing_score,
                phrase_rules=list(range(21)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["fashion"])
        self.assertEqual(autofix_analysis.is_level_3(), Answer(True, AutofixMessage.Doubtful))

    def test_doubtful_when_phrase_score_high(self):
        """
        When the phrase score is high, the Level3 is not valid
        even when the phrase scoring is skewed.
        """

        racing_score = LOW_CONFIDENCE_SCORE_THRESHOLD - 5
        fashion_score = HIGH_CONFIDENCE_SCORE_THRESHOLD
        categories = {
            "fashion": ClassifierCategoryStat(
                score=fashion_score,
                rating=RatingName.BOULDER,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=fashion_score,
                phrase_rules=list(range(14)),
                regex_score=0,
                regex_rules=[],
            ),
            "racing": ClassifierCategoryStat(
                score=racing_score,
                rating=RatingName.STONE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=racing_score,
                phrase_rules=list(range(21)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["fashion"])
        self.assertEqual(autofix_analysis.is_level_3(), Answer(False, AutofixMessage.Unknown))


class TestAutofixAnalysisLevel4(TestCase):
    def test_worst(self):
        paganwiccan_score = LOW_CONFIDENCE_SCORE_THRESHOLD - 5
        fashion_score = HIGH_CONFIDENCE_SCORE_THRESHOLD
        categories = {
            "fashion": ClassifierCategoryStat(
                score=fashion_score,
                rating=RatingName.BOULDER,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=fashion_score,
                phrase_rules=list(range(14)),
                regex_score=0,
                regex_rules=[],
            ),
            "paganwiccan": ClassifierCategoryStat(
                score=paganwiccan_score,
                rating=RatingName.STONE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=paganwiccan_score,
                phrase_rules=list(range(21)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["fashion"])
        self.assertEqual(autofix_analysis.is_level_4(), Answer(True, AutofixMessage.Worst))

    def test_not_doubtful_when_score_skewed(self):
        racing_score = MED_CONFIDENCE_SCORE_THRESHOLD - 97
        violence_score = MED_CONFIDENCE_SCORE_THRESHOLD - 10
        auto_score = MED_CONFIDENCE_SCORE_THRESHOLD - 78
        categories = {
            "violence": ClassifierCategoryStat(
                score=violence_score,
                rating=RatingName.ROCK,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=violence_score,
                phrase_rules=list(range(5)),
                regex_score=0,
                regex_rules=[],
            ),
            "racing": ClassifierCategoryStat(
                score=racing_score,
                rating=RatingName.STONE,
                action=RedwoodAction.Block,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=racing_score,
                phrase_rules=list(range(21)),
                regex_score=0,
                regex_rules=[],
            ),
            "auto": ClassifierCategoryStat(
                score=auto_score,
                rating=RatingName.SILT,
                action=RedwoodAction.Allow,
                domain_score=0,
                domain_rules=[],
                ip_score=0,
                ip_rules=[],
                phrase_score=auto_score,
                phrase_rules=list(range(17)),
                regex_score=0,
                regex_rules=[],
            ),
        }
        category_analysis = CategoryAnalysis(categories)
        rating_analysis = CombinedRatingAnalysis(combine_ratings(categories))
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.top_category_stats, categories["violence"])
        self.assertEqual(autofix_analysis.is_level_4(), Answer(False, AutofixMessage.Unknown))

    def test_monmouth_park_autofix(self):
        category_analysis = CategoryAnalysis(monmouth_park_categories)
        rating_analysis = CombinedRatingAnalysis(monmouth_park_ratings)
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.is_level_1(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(autofix_analysis.is_level_2(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(autofix_analysis.is_level_3(), Answer(True, AutofixMessage.Questionable))
        self.assertEqual(autofix_analysis.is_level_4(), Answer(False, AutofixMessage.Unknown))

    def test_sports_betting_autofix(self):
        category_analysis = CategoryAnalysis(sports_betting_categories)
        rating_analysis = CombinedRatingAnalysis(sports_betting_ratings)
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertTrue(rating_analysis.level_2_rated())

        self.assertEqual(autofix_analysis.is_level_1(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(autofix_analysis.is_level_2(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(autofix_analysis.is_level_3(), Answer(True, AutofixMessage.Confident))
        self.assertEqual(autofix_analysis.is_level_4(), Answer(False, AutofixMessage.Unknown))

    def test_nfl_autofix(self):
        category_analysis = CategoryAnalysis(nfl_categories)
        rating_analysis = CombinedRatingAnalysis(nfl_ratings)
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertTrue(rating_analysis.level_3_rated())

        self.assertEqual(autofix_analysis.is_level_1(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(autofix_analysis.is_level_2(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(autofix_analysis.is_level_3(), Answer(True, AutofixMessage.Confident))
        self.assertEqual(autofix_analysis.is_level_4(), Answer(False, AutofixMessage.Unknown))

    def test_getty_boulder_search_autofix(self):
        category_analysis = CategoryAnalysis(getty_boulder_categories)
        rating_analysis = CombinedRatingAnalysis(getty_boulder_ratings)
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.is_level_1(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(autofix_analysis.is_level_2(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(autofix_analysis.is_level_3(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(autofix_analysis.is_level_4(), Answer(True, AutofixMessage.RequireLevel4))

    def test_bargain_hunter_autofix(self):
        category_analysis = CategoryAnalysis(bargain_hunter_categories)
        rating_analysis = CombinedRatingAnalysis(bargain_hunter_ratings)
        autofix_analysis = AutofixAnalysis(category_analysis, rating_analysis)

        self.assertEqual(autofix_analysis.is_level_1(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(
            autofix_analysis.is_level_2(),
            Answer(True, AutofixMessage.ModerateScoreRatio),
        )
        self.assertEqual(autofix_analysis.is_level_3(), Answer(False, AutofixMessage.Unknown))
        self.assertEqual(autofix_analysis.is_level_4(), Answer(False, AutofixMessage.Unknown))
