from invoke import task, Context
from ipaddress import ip_address
import maxminddb
from ..settings import AUTH_LOG_FIELDS, GEO_IP_DATABASE_DIR
from .utils import line_to_dict


@task(help={"logline": "Redwood auth event log line from auth.log"})
def columns(ctx: Context, logline: str) -> None:  # noqa
    """
    Print out columns of device authentication event and get GeoIP details.
    """
    if not logline:
        raise SystemExit("Provide a valid logline from the auth.log file")

    ll = line_to_dict(logline, AUTH_LOG_FIELDS)
    if not ll:
        return

    print()
    for column, value in ll.items():
        print(f"{column.replace('_', ' ').title()}:  {value or 'N/A'}")
    confirmation_messages(ll)

    geoip_details(ll["client_ip_address"])

    print()


def geoip_details(ip: str) -> None:
    """
    Print GeoIP details about the IP address if it's not private
    """
    try:
        ip = ip.split(":")[0]
        ip_addr = ip_address(ip)
    except (AttributeError, ValueError):
        print(f"Unable to extract GeoIP details for {ip!r}")
        return

    print(f'\nGeoIP data for "{ip}"')
    if ip_addr.is_private:
        print(f'"{ip}" is LAN IP')
        return
    if ip_addr.is_loopback:
        print(f'"{ip}" is loopback IP')
        return

    with maxminddb.open_database(f"{GEO_IP_DATABASE_DIR}/dbip-asn-lite.mmdb") as reader:
        if asn := reader.get(ip):
            try:
                print(
                    "ASN: {autonomous_system_number}\n"
                    "Organization: {autonomous_system_organization}".format(**asn)  # type: ignore[arg-type]
                )
            except KeyError:
                print(f"ASN is {asn}")

    with maxminddb.open_database(f"{GEO_IP_DATABASE_DIR}/dbip-city-lite.mmdb") as reader:
        if cc := reader.get(ip):
            city = get_english_name("city", cc)  # type: ignore[arg-type]
            state = get_english_name("subdivisions", cc)  # type: ignore[arg-type]
            country = get_english_name("country", cc)  # type: ignore[arg-type]
            print(f"Location: {city}, {state}, {country}")


def get_english_name(key: str, data: dict[str, dict | list[dict]]) -> str:
    """
    Get the English name for the key in question.

    cc = {
        'city': {'names': {'en': 'Hudson'}},
        'continent': {'code': 'NA', 'geoname_id': 6255149, 'names': {'de': 'Nordamerika', 'en': 'North America', 'es': 'Norteamérica', 'fa': ' امریکای شمالی', 'fr': 'Amérique Du Nord', 'ja': '北アメリカ大陸', 'ko': '북아메리카', 'pt-BR': 'América Do Norte', 'ru': 'Северная Америка', 'zh-CN': '北美洲'}},
        'country': {'geoname_id': 6252001, 'is_in_european_union': False, 'iso_code': 'US', 'names': {'de': 'Vereinigte Staaten von Amerika', 'en': 'United States', 'es': 'Estados Unidos de América (los)', 'fa': 'ایالات متحدهٔ امریکا', 'fr': 'États-Unis', 'ja': 'アメリカ合衆国', 'ko': '미국', 'pt-BR': 'Estados Unidos', 'ru': 'США', 'zh-CN': '美国'}},
        'location': {'latitude': 42.3918, 'longitude': -71.5662},
        'subdivisions': [{'names': {'en': 'Massachusetts'}}, {'latitude': 42.3918, 'longitude': -71.5662}],
    }
    """
    try:
        names = data[key]
    except KeyError:
        return ""

    if isinstance(names, list):
        for row in names:
            if "names" in row:
                names = row
                break
        else:
            return ""

    return names["names"].get("en") or ""


def confirmation_messages(ll: dict[str, str]) -> None:
    """
    Print a message to the user suggesting that they verify
    that the device details are correctly specified.
    """
    if ll["credential_status"] != "correct" or ll["auth_type"] == "acl exception":
        if ll["user_agent"] and ll["network"]:
            print(
                "\nConfirm that this device is configured "
                "with correct Platform and Expected Networks."
            )
