from tabulate import tabulate

from ..redwood.categories import sort_categories
from ..typehints import ClassifyText, TALLY_RESPONSES, RatingName, RatingStat


def print_verbose_results(tally: TALLY_RESPONSES | ClassifyText) -> None:
    """
    Print human-friendly output of classifier
    output of "verbose" method.

    results = {
        'text': '<classified text>',
        'categories': {
            'ag': 10,
            'computer': 20,
            'news/news_sports': 44,
        },
        'rules': {
            '< baseball >': 1,
            '< football>': 1,
            '< repository>': 1,
        },
    }
    """
    rules = tally.rules
    category_table = [(cat, score) for cat, score in tally.categories.items()]

    print(f"\n{len(tally.categories)} Categories Found")
    print(tabulate(category_table, headers=["Category Code", "Score"], tablefmt="fancy_outline"))

    print(f"\n{len(rules)} Rules Found\n------------------------------")
    for rule, score in rules.items():
        print(f"{rule} {score}")

    print()


def print_score_analysis(tally: TALLY_RESPONSES, verbose: bool = False) -> None:
    """
    Print human-friendly output of classifier
    output of "analyze-score" method.
    """
    score_analysis = tally.scoreAnalysis
    scoring_categories = tally.categories

    print_analyze_score_category_table("Scoring Categories", scoring_categories, tally)

    if verbose:
        low_scorers = {cat: 0 for cat in score_analysis.keys() if cat not in scoring_categories}

        for category in low_scorers:
            for rule, stats in score_analysis[category].items():
                low_scorers[category] += stats.Score

        low_scorers = sort_categories(low_scorers)
        name = "ACL Categories or Low-scoring Categories"
        print_analyze_score_category_table(name, low_scorers, tally)


def print_analyze_score_category_table(
    name: str,
    categories: dict[str, int],
    tally: TALLY_RESPONSES,
) -> None:
    """
    Print the category table with score and rule counts.
    """
    print(f"\n{len(categories)} {name}")

    headers = ["Category Code", "Score", "Rules"]
    category_table = []
    score_analysis = tally.scoreAnalysis

    for category, score in categories.items():
        category_table.append((category, score, len(score_analysis[category])))

    print(tabulate(category_table, headers=headers, tablefmt="fancy_outline"))

    print_category_stats(categories, tally)


def print_category_stats(categories: dict[str, int], tally: TALLY_RESPONSES) -> None:
    """
    Print the categories and the rules in each one.
    """
    score_analysis = tally.scoreAnalysis
    for category, score in categories.items():
        print(
            "\n----------------------------------------------\n"
            f"# {category} ({score})\n"
            "----------------------------------------------"
        )
        rules = []
        for rule, stats in score_analysis[category].items():
            rule_count = stats.Count
            rule_score = stats.Score
            rules.append((rule, rule_score, f"({rule_count} * {rule_score // rule_count})"))

        rules.sort(key=lambda r: r[1], reverse=True)
        print(tabulate(rules, tablefmt="plain"))

    print()


def print_combined_ratings(ratings: dict[RatingName, RatingStat]) -> None:
    """
    Print combined ratings to the shell.
    """
    print("Combined Ratings")
    rating_table = [(r, s.total_score, s.phrase_score) for r, s in ratings.items()]
    sorted_ratings = sorted(rating_table, key=lambda r: r[1], reverse=True)
    headers = ["Name", "Total Score", "Phrase Score"]
    print(tabulate(sorted_ratings, headers=headers, tablefmt="fancy_outline"))
    print()


__all__ = (
    "print_category_stats",
    "print_verbose_results",
    "print_score_analysis",
    "print_combined_ratings",
)
