from lchttp.yamltools import yaml_safe_load
import os
from pathlib import Path
from typing import NamedTuple

from redwoodctl.settings import (
    NON_SCORING_CATEGORIES,
    LC_CAT_PREFIX,
    PRUNE_CATEGORIES,
    REDWOOD_CATEGORY_DIR,
)
from redwoodctl.typehints import (
    RedwoodAction,
    CategoryConf,
    CATEGORY_CONFIGS,
    CATEGORY_NAMES_SCORES,
    ClassifyText,
    ClassifierCategoryStat,
    EXTENDED_RESPONSES,
    NUMBER,
    TALLY_RESPONSES,
    NON_SCORING_RATINGS,
)
from .tally import rule_type


class TotalScore(NamedTuple):
    total_score: int
    phrase_score: int


def add_scoring_category_scores(tally: TALLY_RESPONSES) -> CATEGORY_NAMES_SCORES:
    """
    For endpoints that don't provide the "categories" key,
    calculate and include Scoring Category scores key to
    the output of the Tally analysis.

    'categories': {
        'audioelectronics': 268,
        'computer': 220,
    }
    """
    score_analysis = tally.scoreAnalysis
    categories = {cat: 0 for cat in score_analysis.keys()}

    for category, score in categories.items():
        for rule, stats in score_analysis[category].items():
            categories[category] += stats.Score

    return sort_categories({cat: score for cat, score in categories.items() if score > 200})


def add_classifier_rule_analysis(
    tally: TALLY_RESPONSES,
    category_configs: CATEGORY_CONFIGS,
) -> dict[str, ClassifierCategoryStat]:
    """
    Calculate the total number of each Rule Type and Score that
    each type contributed to the final Classifier Category score.

    Only interested in Classifier Categories here!! ACL and Misc
    categories should not be included!
    """
    category_rules = {}

    for category in tally.categories:
        conf = category_configs.get(category)
        if not conf:
            continue

        rules = tally.scoreAnalysis.get(category) or {}

        try:
            if (action := conf.action) == RedwoodAction.ACL:
                continue
            if (rating := conf.rating) in NON_SCORING_RATINGS:
                continue
        except AttributeError:
            continue

        cat_stats = ClassifierCategoryStat(
            score=0,
            rating=rating,
            action=action,
            domain_score=0,
            domain_rules=[],
            ip_score=0,
            ip_rules=[],
            phrase_score=0,
            phrase_rules=[],
            regex_score=0,
            regex_rules=[],
        )

        for rule, stats in rules.items():
            type_prefix = rule_type(rule)
            cat_stats.increment(f"{type_prefix}_score", stats.Score)
            cat_stats.append(f"{type_prefix}_rules", rule)
            cat_stats.score += stats.Score

        category_rules[category] = cat_stats

    return category_rules


def load_category_conf_files() -> CATEGORY_CONFIGS:
    """
    Load category.conf files into basic config dictionary.
    """
    category_config_files = {}
    os.chdir(REDWOOD_CATEGORY_DIR)

    for category in Path(".").rglob("*/category.conf"):
        with open(category) as cc:
            data = yaml_safe_load(cc.read())
            try:
                conf = CategoryConf(
                    id=data["id"],
                    action=data["action"],
                    genre_id=data["genre_id"],
                    rating=data["grade"],
                    parent_multiplier=data.get("parent_multiplier") or 1.0,
                )
            except KeyError:
                continue
            category_config_files["/".join(category.parts[:-1])] = conf

    return category_config_files


def prune_child_categories(
    tally: TALLY_RESPONSES | ClassifyText,
    category_configs: CATEGORY_CONFIGS | None = None,
) -> EXTENDED_RESPONSES:
    """
    Remove child categories that have scores equal or lower than their
    Parent Category * multiplier, since they provided no intelligence of their own.

    results = {
        'url': 'https://www.bing.com:443/?toWww=1&redig=6F1A2AD5820E420EAF4848E7F0431FCA',
        'categories': {'audioelectronics': 220, 'computer': 267, 'searchengines': 200},
        'scoreAnalysis': {
            'ag': {
                '<gallo>': {'Count': 1, 'Score': 20},
            },
            'audioelectronics': {
                '<microphone>': {'Count': 11, 'Score': 220},
            },
            'computer': {
                '< address bar>': {'Count': 1, 'Score': 15},
                '< games>': {'Count': 5, 'Score': -50},
                '<ajax request>': {'Count': 1, 'Score': 20},
                '<bugfix>': {'Count': 1, 'Score': 15},
                '<datasource>': {'Count': 1, 'Score': 27},
                '<desktop app>': {'Count': 1, 'Score': 25},
                '<microsoft 365>': {'Count': 3, 'Score': 75},
                '<onenote>': {'Count': 7, 'Score': 70},
                '<powerpoint>': {'Count': 7, 'Score': 70},
            },
            'searchengines': {
                'bing.com': {'Count': 1, 'Score': 200},
            },
        },
    }
    """
    categories = tally.categories

    # Remove Console Categories
    for category in list(categories.keys()):
        if category.startswith(LC_CAT_PREFIX) or category in PRUNE_CATEGORIES:
            categories.pop(category)

    for category in NON_SCORING_CATEGORIES:
        categories.pop(category, None)

    if not categories:
        return tally

    categories = sort_categories(tally.categories)
    category_configs = category_configs or load_category_conf_files()

    parent_categories = {}
    child_categories = {}

    for category, score in categories.items():
        if "/" in category:
            child_categories[category] = score
        else:
            parent_categories[category] = score

    for child_category, child_score in child_categories.items():
        parent = child_category[: child_category.find("/")]

        try:
            parent_score = categories[parent]
            parent_multiplier = category_configs[parent].parent_multiplier
            if child_score <= parent_score * parent_multiplier:
                categories.pop(child_category, None)
        except KeyError:
            pass

    tally.categories = categories

    return tally


def prune_score_analysis_categories(tally: TALLY_RESPONSES) -> EXTENDED_RESPONSES:
    """
    Remove categories from `scoreAnalysis` that aren't in the `categories`
    key, since they're not contributing to the Score.

    When performing autofix analysis, especially, this data should be purged,
    so we don't need to worry about non-scoring data there.
    """
    scrub = []

    if not (scorers := tally.categories):
        return tally

    for category in tally.scoreAnalysis:
        if category not in scorers:
            scrub.append(category)

    for category in scrub:
        tally.scoreAnalysis.pop(category, None)

    return tally


def sort_categories(categories: CATEGORY_NAMES_SCORES) -> CATEGORY_NAMES_SCORES:
    """
    Sort categories in descending order by score for easier presentation.
    """
    category_list = [(cat, score) for cat, score in categories.items()]
    return dict(sorted(category_list, key=lambda c: c[1], reverse=True))


def calculate_total_score(tally: TALLY_RESPONSES) -> TotalScore:
    """
    Calculate combined score of all scoring Classifier Categories.
    """
    total_score: NUMBER = 0
    phrase_score: NUMBER = 0

    for category, stats in tally.classifierAnalysis.items():
        total_score += stats.score
        phrase_score += stats.phrase_score

    return TotalScore(int(total_score), int(phrase_score))


__all__ = (
    "add_scoring_category_scores",
    "add_classifier_rule_analysis",
    "load_category_conf_files",
    "prune_child_categories",
    "prune_score_analysis_categories",
    "sort_categories",
    "calculate_total_score",
)
