from typing import NamedTuple
from redwoodctl.typehints import (
    AutofixTier,
    AutofixMessage,
    AutofixMode,
    ClassifierCategoryStat,
    CategoryCodeStats,
    RatingStat,
    RatingCodeStats,
    RatingName,
    RedwoodAction,
    RatingRatio,
)
from ..settings import (
    PERMIT_LEVEL_TWO_AUTOFIX,
    REQUIRE_LEVEL_THREE_AUTOFIX,
    MED_CONFIDENCE_SCORE_THRESHOLD,
    AUTOFIXABLE_BOULDER_CATEGORIES,
)

# When transition ratings are found with more positive ratings,
# then it presumably is not unduly objectionable.
# But if it's the ONLY rating, or found only in the presence of more
# objectionable ratings, then it doesn't provide any elevating of tone.
LEVEL_2_TRANSITION = frozenset([RatingName.SAND])
LEVEL_1_BASE = frozenset((RatingName.BASE, RatingName.SILT))
LEVEL_1_RATINGS = LEVEL_1_BASE.union(LEVEL_2_TRANSITION)

LEVEL_2_RATINGS = frozenset((RatingName.SAND, RatingName.PEBBLE))
LEVEL_1_and_2_RATINGS = LEVEL_1_RATINGS.union(LEVEL_2_RATINGS)

LEVEL_3_RATINGS = frozenset([RatingName.STONE])
LEVEL_1_to_3_RATINGS = LEVEL_1_and_2_RATINGS.union(LEVEL_2_RATINGS)

LEVEL_4_RATINGS = frozenset((RatingName.ROCK, RatingName.BOULDER, RatingName.STONE))


class Answer(NamedTuple):
    result: bool
    message: AutofixMessage


class Level(NamedTuple):
    name: AutofixTier
    message: AutofixMessage
    action: AutofixMode


class CombinedRatingAnalysis:
    """
    Analyze the Ratings in this Tally Response.
    """

    def __init__(self, ratings: RatingCodeStats):
        self.combined_score = sum(rs.total_score for rs in ratings.values())
        self.combined_phrase_score = sum(rs.phrase_score for rs in ratings.values())
        self.ratings = ratings
        self._top_rating: RatingName | str = ""
        self._top_rating_stats: RatingStat | None = None
        # Ratings below 10% of the total will not be included
        self.rating_names = frozenset(
            r for r, stats in ratings.items() if stats.confidence(self.combined_score) >= 0.1
        )
        self.rating_confidence, self.phrase_confidence = self.calculate_rating_ratios()

    def calculate_rating_ratios(self) -> tuple[RatingRatio, RatingRatio]:
        """
        Combine all ratings into Level1-4, based on the Classifier-scoring.
        This is to give weight to the phrase-based classifying process when
        comparing the combined ratings on a given request.
        """
        combined_rating_confidence = RatingRatio()
        combined_phrase_confidence = RatingRatio()

        for rating, stats in self.ratings.items():
            rating_confidence = stats.confidence(self.combined_score)
            phrase_confidence = stats.phrase_confidence(self.combined_phrase_score)

            if rating in LEVEL_1_BASE:
                combined_rating_confidence.Level1 += rating_confidence
                combined_phrase_confidence.Level1 += phrase_confidence
            elif rating in LEVEL_2_RATINGS:
                combined_rating_confidence.Level2 += rating_confidence
                combined_phrase_confidence.Level2 += phrase_confidence
            elif rating in LEVEL_3_RATINGS:
                combined_rating_confidence.Level3 += rating_confidence
                combined_phrase_confidence.Level3 += phrase_confidence
            elif rating in LEVEL_4_RATINGS:
                combined_rating_confidence.Level4 += rating_confidence
                combined_phrase_confidence.Level4 += phrase_confidence

            # Track the presence of any Boulder-rated categories for convenience.
            if rating == RatingName.BOULDER:
                combined_rating_confidence.BOULDER += rating_confidence
                combined_phrase_confidence.BOULDER += phrase_confidence

        return combined_rating_confidence, combined_phrase_confidence

    def top(self) -> RatingName:
        """
        Return the rating with the highest total score.
        """
        if not self._top_rating:
            ratings = [(rating, stats) for rating, stats in self.ratings.items()]
            if len(ratings) > 1:
                ratings.sort(key=lambda r: r[1].total_score, reverse=True)

            try:
                self._top_rating = ratings[0][0]
            except IndexError:
                pass

        return self._top_rating

    def stats(self) -> RatingStat:
        """
        Return top rating Stats, but don't crash if there were no ratings.
        """
        if not self._top_rating_stats:
            try:
                self._top_rating_stats = self.ratings[self.top()]
            except KeyError:
                self._top_rating_stats = RatingStat.empty()
        return self._top_rating_stats

    def level_1_rated(self) -> bool:
        """
        Combined ratings have no negative content.
        """
        # Level1-2 transition ratings count as Level1 if some phrase scoring occurred
        if self.rating_names == LEVEL_2_TRANSITION:
            return not bool(self.stats().phrase_score)

        if LEVEL_1_RATINGS.issuperset(self.rating_names):
            return True

        return (
            self.rating_confidence.Level1 >= 0.5
            and not self.rating_confidence.Level3
            and not self.rating_confidence.Level4
        )

    def level_2_rated(self) -> bool:
        """
        Combined ratings have none of the worst ratings. Suitable for Level 2 Autofix.
        Much of the time, Autofixes should fit into Level2.
        """

        # Level1-2 transition ratings count as Level2 if phrase scoring occurred
        if self.rating_names == LEVEL_2_TRANSITION:
            return bool(self.stats().phrase_score)

        if LEVEL_2_RATINGS.issuperset(self.rating_names):
            return True

        rating_ratios = self.rating_confidence
        if LEVEL_3_RATINGS.issuperset(self.rating_names):
            return self.phrase_confidence.Level3 < 0.5 or self.stats().phrase_scoring_skewed()

        if not rating_ratios.Level4:
            # Some of Levels 1 & 2 still count as Level 2
            if rating_ratios.Level3 and (rating_ratios.Level1 or rating_ratios.Level2):
                return True
            return rating_ratios.Level2 > rating_ratios.Level1

        if rating_ratios.Level1 + rating_ratios.Level2 > 0.7 and not rating_ratios.BOULDER:
            return True

        return rating_ratios.Level4 <= 0.075

    def level_3_rated(self) -> bool:
        """
        A narrow slice between Level2 and Level4. Level3 should be autofix-able if
        some positive content is found, and no boulder content is found.
        """
        ratios = self.rating_confidence
        levels_one_two = ratios.Level1 + ratios.Level2

        if ratios.Level4:
            if levels_one_two > ratios.Level3 + ratios.Level4:
                # If no boulder-rated categories are found, it can be Level2.
                return bool(ratios.BOULDER)
        else:
            if self.stats().phrase_scoring_skewed() or ratios.Level3 < 0.55:
                return False

        if not ratios.Level3:
            return False

        # Level3 can be Level2 if the phrase scoring confidence is low.
        if levels_one_two < 0.3 and self.phrase_confidence.Level3 >= 0.5:
            return True

        # If there's some good content, then it might be worth the risk
        if levels_one_two >= 0.3 and ratios.Level4 <= 0.5:
            return True

        return levels_one_two + ratios.Level3 >= 0.45 and ratios.Level4 <= 0.35

    def level_4_rated(self) -> bool:
        """
        Combined ratings include no positive content.
        """
        ratios = self.rating_confidence

        if round((level_4 := ratios.Level4), 2) < 0.075:
            return False

        if LEVEL_4_RATINGS.issuperset(self.rating_names):
            return True

        if not ratios.Level1 and not ratios.Level2 and ratios.Level3 and ratios.Level4:
            return True

        return level_4 >= 0.5 or self.rating_confidence.Level3 + level_4 >= 0.8


class CategoryAnalysis:
    """
    Analyze the Categories in this Tally Response.
    """

    def __init__(self, categories: CategoryCodeStats):
        self.categories = categories
        self._top_category: str = ""
        self._top_category_stats: ClassifierCategoryStat | None = None

    def top(self) -> str:
        """
        Return the top-scoring category, if one was found.
        """
        if not self._top_category and self.categories:
            cats = [(cat, stats) for cat, stats in self.categories.items()]
            if len(cats) > 1:
                cats.sort(key=lambda r: r[1].score, reverse=True)

            try:
                self._top_category = cats[0][0]
            except IndexError:
                pass

        return self._top_category

    def stats(self) -> ClassifierCategoryStat:
        if not self._top_category_stats:
            try:
                self._top_category_stats = self.categories[self.top()]
            except KeyError:
                self._top_category_stats = ClassifierCategoryStat.empty()

        return self._top_category_stats

    def level_1_rated(self) -> bool:
        """
        Category is silt-rated, and not a concern for objectionable content.
        """
        top_cat_stats = self.stats()
        if top_cat_stats.rating in LEVEL_1_BASE:
            return True

        return (
            top_cat_stats.rating == RatingName.SAND and top_cat_stats.action == RedwoodAction.Allow
        )

    def level_2_rated(self) -> bool:
        """
        Category rating isn't from the worst ratings. Suitable for Level 2 Autofix.
        """
        if self.level_1_rated():
            return False

        return self.level_2_permitted() or self.stats().rating in LEVEL_2_RATINGS

    def level_2_permitted(self) -> bool:
        """
        These stone categories may be Level2
        even if phrase scoring is confident.
        """
        return self.top() in PERMIT_LEVEL_TWO_AUTOFIX

    def level_3_rated(self) -> bool:
        """
        Level 3 is a narrow slice between Level2 and Level4,
        and is expected to be disabled most of the time.

        For those who enable it, the possibility of negative content
        is quite high. But it's possibly useful for temporarily
        allowing the worst categories if the scoring was skewed.
        """
        if (stats := self.stats()).rating not in (RatingName.BOULDER, RatingName.ROCK):
            return False

        return self.require_level_three() or self._phrase_scoring_both_low_and_skewed(stats)

    def require_level_three(self) -> bool:
        """
        These categories must be Level3 even if phrase scoring is skewed.
        """
        return self.top() in REQUIRE_LEVEL_THREE_AUTOFIX

    def level_4_rated(self) -> bool:
        """
        Is category too coarse for Allowing?
        """
        if self.require_level_four():
            return True

        stats = self.stats()

        if self._phrase_scoring_both_low_and_skewed(stats):
            return False

        return stats.rating in (RatingName.BOULDER, RatingName.ROCK)

    def require_level_four(self) -> bool:
        """
        The categories with the least margin for error in
        autofixing. They should be Level4 in all cases
        except in lowest of low scoring confidence.
        """
        return (
            self.top() not in AUTOFIXABLE_BOULDER_CATEGORIES
            and self.stats().rating == RatingName.BOULDER
        )

    def _phrase_scoring_both_low_and_skewed(self, stats: ClassifierCategoryStat) -> bool:
        """
        If phrase scoring is low, and skewed, then
        we  want Level 3 rather than Level 4.
        """
        if not stats.phrase_scoring_skewed():
            return False

        # Only consider the score as skewed if the overall score is low,
        # and no request-based rules were found.
        request_based_rules = any((stats.domain_score, stats.ip_score, stats.regex_score))
        return not request_based_rules and stats.phrase_score < MED_CONFIDENCE_SCORE_THRESHOLD


class AutofixAnalysis:
    """
    Analyze this Tally Response for use in Autofix Permission checks, by
    comparing the Stats of the Combined Ratings and Top-scoring Category.
    """

    def __init__(self, categories: CategoryAnalysis, ratings: CombinedRatingAnalysis):
        self.categories = categories
        self.combined_ratings = ratings
        self.top_category = categories.top()
        self.top_category_stats = self.categories.stats()

    def is_level_1(self) -> Answer:
        """
        Level 1 should contain only material that would not
        be objectionable in any context (family / school / work).
        """
        if not self.combined_ratings.level_1_rated():
            return Answer(False, AutofixMessage.Unknown)

        if (scoring_skewed := self.phrase_scoring_skewed()).result:
            return scoring_skewed

        if self.categories.level_1_rated():
            return Answer(True, AutofixMessage.WellRated)

        if self.categories.level_2_rated():
            return Answer(True, AutofixMessage.Satisfactory)

        return Answer(False, AutofixMessage.Unknown)

    def is_level_2(self) -> Answer:
        """
        Level 2 should contain only material that would not
        be objectionable in any recreational context.
        """
        if self.categories.level_2_permitted():
            return Answer(True, AutofixMessage.PermitLevel2)

        if self.combined_ratings.level_2_rated():
            if self.categories.level_1_rated():
                return Answer(True, AutofixMessage.HighScoreRatio)

            if self.categories.level_2_rated():
                return Answer(True, AutofixMessage.ModerateScoreRatio)

        return Answer(False, AutofixMessage.Unknown)

    def is_level_3(self) -> Answer:
        """
        For the most part, the answer should be Level2 or Level4,
        but Level3 exists to consider all the ratings together.
        If the content is likely negative, but does contain at
        least _some_ positive content, consider it Level3.

        Most of the time, autofixing this level will be disabled anyway.
        """
        if self.categories.require_level_three():
            return Answer(True, AutofixMessage.RequireLevel3)

        if self.categories.require_level_four():
            return Answer(False, AutofixMessage.Unknown)

        if self.combined_ratings.level_3_rated():
            if self.combined_ratings.phrase_confidence.Level3 >= 0.6:
                return Answer(True, AutofixMessage.Confident)
            return Answer(True, AutofixMessage.Questionable)

        if self.categories.level_3_rated():
            return Answer(True, AutofixMessage.Doubtful)

        return Answer(False, AutofixMessage.Unknown)

    def is_level_4(self) -> Answer:
        """
        Level 4 is extremely negative / objectionable.
        Autofixing Level 4 cannot be enabled.
        """
        if self.categories.require_level_four():
            return Answer(True, AutofixMessage.RequireLevel4)

        if self.categories.level_3_rated():
            ratios = self.combined_ratings.rating_confidence
            if ratios.Level1 + ratios.Level2 + ratios.Level3 > ratios.Level4:
                return Answer(False, AutofixMessage.Unknown)

        if self.combined_ratings.level_4_rated():
            return Answer(True, AutofixMessage.Worst)

        return Answer(False, AutofixMessage.Unknown)

    def phrase_scoring_skewed(self) -> Answer:
        """
        Calculate if phrase score was unduly influenced by
        a small number of phrases occurring frequently.
        """
        if self.top_category_stats.phrase_scoring_skewed():
            if self.top_category_stats.score < MED_CONFIDENCE_SCORE_THRESHOLD:
                return Answer(True, AutofixMessage.LowCount)
            return Answer(True, AutofixMessage.Skewed)
        return Answer(False, AutofixMessage.Unknown)

    def level(self) -> Level:
        """
        Return the Autofix Permission level required to enable
        autofixing the request that generated this Tally.
        """
        if (answer := self.is_level_1()).result:
            return Level(AutofixTier.Level1, answer.message, AutofixMode.Allow)

        if (answer := self.is_level_2()).result:
            return Level(AutofixTier.Level2, answer.message, AutofixMode.Offset)

        if (answer := self.is_level_3()).result:
            return Level(AutofixTier.Level3, answer.message, AutofixMode.Offset)

        if (answer := self.is_level_4()).result:
            return Level(AutofixTier.Level4, answer.message, AutofixMode.Block)

        return Level(AutofixTier.Level2, AutofixMessage.Unknown, AutofixMode.Offset)


__all__ = (
    "CombinedRatingAnalysis",
    "CategoryAnalysis",
    "AutofixAnalysis",
)
