from invoke.context import Context
from invoke.tasks import task

from redwoodctl.http.redwood import (
    classify_tally as _classify_tally,
    classify_text as _classify_text,
    classify_analyze_url as _classify_analyze_url,
)
from redwoodctl.http.autofix import classify_autofix_url
from redwoodctl.redwood.display import (
    print_score_analysis,
    print_verbose_results,
    print_combined_ratings,
)
from redwoodctl.typehints.redwood_api_types import ApiError


@task(
    name="text",
    help={
        "text": "The text string for Redwood to download and classify.",
    },
)
def classify_text(ctx: Context, text: str) -> None:
    """Pass the specified text to Redwood to classify."""
    response = _classify_text(text)
    if isinstance(response, ApiError):
        raise SystemExit(f"\nUnable to classify text. Error message: {response.message}.\n")
    print_verbose_results(response)


@task(
    name="tally",
    help={
        "tally": "The tally string for Redwood to parse and analyze.",
    },
)
def classify_tally(ctx: Context, tally: str) -> None:
    """Pass the specified tally to Redwood to classify."""
    response = _classify_tally(tally)
    if isinstance(response, ApiError):
        raise SystemExit(f"\nUnable to classify tally. Error message: {response.message}.\n")
    print_score_analysis(response)


@task(
    name="url",
    help={
        "url": "The URL for Redwood to download and classify.",
        "verbose": "Print all categories including those below the threshold.",
    },
)
def classify_url(ctx: Context, url: str, verbose: bool = False) -> None:
    """
    Download and classify the specified URL.
    """
    response = _classify_analyze_url(url, verbose=verbose)
    if isinstance(response, ApiError):
        raise SystemExit(f"\nUnable to classify URL. Error message: {response.message}.\n")

    print(f"\nURL: {url}")
    print_score_analysis(response, verbose)


@task(
    name="autofix",
    help={
        "url": "The URL that might be autofixed.",
    },
)
def autofix_level(ctx: Context, url: str) -> None:
    """
    Determine the permission level that would be required
    to autofix the specified URL.
    """
    response, analysis = classify_autofix_url(url)

    if isinstance(response, ApiError):
        raise SystemExit(f"\nUnable to classify URL. Error message: {response.message}.\n")

    if not analysis:
        raise SystemExit("\nAn unknown error occurred.\n")

    print(f"\nURL: {url}")
    level = analysis.level()
    print(
        f"\nCategory: {analysis.top_category}"
        f"\nRating: {analysis.top_category_stats.rating}"
        f"\nLevel: {level.name.label}"
        f"\nMessage: {level.message.label}"
        f"\nAction: {level.action.label}\n"
    )

    print_combined_ratings(response.ratings)


__all__ = (
    "classify_tally",
    "classify_text",
    "classify_url",
    "autofix_level",
)
