from .mixins import CRUD


class PowerDNSRecord:
    """
    Base class for CRUD operations on PowerDNS Records.

    Subclasses class must define `path`.

    path: Base URL path on which requests will be sent (e.g. 'uapi/devices')
    """
    path = None

    def __init__(self, powerdns):
        """
        :param powerdns: `powerdns.PowerDNS` connection to make requests
        """
        self.powerdns = powerdns


class Servers(CRUD, PowerDNSRecord):
    path = 'servers'

    def base_path(self, path=''):
        base = super().base_path(path)
        return f'{base}/{self.powerdns.server_id}'


class Zones(Servers):
    path = 'servers'

    def base_path(self, path=''):
        base = super().base_path(path)
        return f'{base}/zones'

    def update(self, oid, data: dict):
        """
        Create a new record set in the domain zone.

        :param data: Data to send to the server to create the resource
        """
        return self.http_patch(oid=oid, post_data=data)


class PDNSAdmin(CRUD, PowerDNSRecord):
    path = 'pdnsadmin'


class AdminZones(CRUD, PowerDNSRecord):
    path = 'pdnsadmin/zones'


class Accounts(CRUD, PowerDNSRecord):
    path = 'pdnsadmin/accounts'


class AccountsUsers(CRUD, PowerDNSRecord):
    path = 'pdnsadmin/accounts/users'


class APIKeys(CRUD, PowerDNSRecord):
    path = 'pdnsadmin/apikeys'


class Users(CRUD, PowerDNSRecord):
    path = 'pdnsadmin/users'
