from lcrequests import Request

from . import records
from .exceptions import *


class PowerDNS:
    """
    PowerDNS Admin API Class.
    Represent a connection to a PowerDNS Admin server.
    """

    def __init__(self, token, server_url='https://dns.compassfoundation.io/api/v1'):
        """
        :param server_url: Base URL for PowerDNS server. Example: https://dns.example.com/api/v1
        """
        self.api_key = token
        self.url = server_url

        self.servers = records.Servers(self)
        self.apikeys = records.APIKeys(self)
        self.accounts = records.Accounts(self)
        self.accountsusers = records.AccountsUsers(self)
        self.users = records.Users(self)
        self.zones = records.Zones(self)
        self.pdnsadmin = records.PDNSAdmin(self)
        self.adminzones = records.AdminZones(self)

        self.headers = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
            'X-API-Key': self.api_key,
        }

        self.server_id = 'localhost'

    def load_server(self, server_id):
        self.server_id = server_id

    def http_request(self, verb: str, url: str, headers: dict = None, post_data: dict = None):
        """Make an HTTP request to the PowerDNS server.

        :param verb: The HTTP method to call ('get', 'post', 'put', 'delete')
        :param url: Full URL to query (/servers, or /servers/<server_id>)
        :param headers: HTTP headers to add to default headers
        :param post_data: Data to send in the body (will be converted to json)

        Returns a request result object.
        """
        default_headers = self.headers.copy()
        headers = headers or {}
        default_headers.update(headers)

        response = Request(
            url=url,
            headers=default_headers,
        ).request(
            method=verb,
            url=url,
            data=post_data,
        )

        if not response.ok:
            try:
                msg = ','.join(response.json()['errors'])
            except (ValueError, TypeError, KeyError):
                if response.reason:
                    msg = response.reason
                else:
                    msg = f'{response.status_code}: {response.text}'

            if response.status_code == 401:
                raise PowerDNSAuthenticationError(msg, response=response)

            if response.status_code == 403:
                raise PowerDNSAuthenticationError(msg, response=response)

            if 400 <= response.status_code < 500:
                raise PowerDNSClientError(msg, response=response)

            raise PowerDNSServerError(
                f'{msg} {response.status_code}', response=response
            )

        if response.status_code == 302:
            raise PowerDNSError('Unexpected Redirect', response=response)

        return response

    def __str__(self):
        return f'PowerDNS API - Server: {self.url}'

    def __repr__(self):
        return f'PowerDNS(token="<token>" server_url={self.url!r})'


__all__ = [
    'PowerDNS',
]
