DELETE = 'DELETE'
GET = 'GET'
POST = 'POST'
PUT = 'PUT'
PATCH = 'PATCH'


class Base:

    def base_path(self, path: str = ''):
        """
        Hook to generate base path for a given model.
        """
        return path or self.path

    def build_url(self, path: str ='', oid=None):
        """
        Generate URL path, supporting custom API routes.

        :param path: Path to resource (/servers, or /pdnsadmin/zones)
        :param oid: Object ID such as domain name or zone name
        """
        url = f'{self.powerdns.url}/{self.base_path(path)}'
        if oid:
            return f'{url}/{oid}'
        return url

    def http_list(self, path: str = ''):
        """Make a GET request to the PowerDNS server.

        :param path: Path to resource (/servers, or /zones)

        Returns the parsed json returned by the server.
        """
        return self.powerdns.http_request(GET, self.build_url(path))

    def http_get(self, path: str = '', oid=None):
        """Make a GET request to the PowerDNS server.

        :param path: Path to resource (/servers, or /pdnsadmin/zones)
        :param oid: Object ID such as domain name or zone name

        Returns the parsed json returned by the server.
        """
        return self.powerdns.http_request(GET, self.build_url(path, oid))

    def http_post(self, path: str = '', oid=None, post_data=None):
        """Make a POST request to the PowerDNS server.

        :param path: Path to resource (/servers, or /pdnsadmin/zones)
        :param oid: Object ID such as domain name or zone name
        :param post_data: Data to send in the body (will be converted to json)

        Returns the parsed json returned by the server.
        """
        return self.powerdns.http_request(
            verb=POST,
            url=self.build_url(path, oid),
            post_data=post_data or {},
        )

    def http_put(self, path: str = '', oid=None, post_data=None):
        """Make a PUT request to the PowerDNS server.

        :param path: Path to resource (/servers, or /pdnsadmin/zones)
        :param oid: Object ID
        :param post_data: Data to send in the body (will be converted to json)

        Returns the parsed json returned by the server.
        """
        return self.powerdns.http_request(
            verb=PUT,
            url=self.build_url(path, oid),
            post_data=post_data or {},
        )

    def http_patch(self, path: str = '', oid=None, post_data: dict = None):
        """Make a PATCH request to the PowerDNS server.

        :param path: Path to resource (/servers, or /pdnsadmin/zones)
        :param oid: Object ID such as domain name or zone name
        :param post_data: Data to send in the body (will be converted to json)

        Returns the parsed json returned by the server.
        """
        return self.powerdns.http_request(
            verb=PATCH,
            url=self.build_url(path, oid),
            post_data=post_data or {},
        )

    def http_delete(self, path: str = '', oid=None, **kwargs):
        """Make a DELETE request to the PowerDNS server.

        :param path: Path to resource (/servers, or /pdnsadmin/zones)
        """
        return self.powerdns.http_request(DELETE, self.build_url(path, oid))


class Create:

    def create(self, oid, data: dict):
        """
        Create a new record.

        :param data: Data to send to the server to create the resource
        :param kwargs: Extra options to send to the server
        :return: dictionary of record
        """
        return self.http_post(oid=oid, post_data=data)


class Get:

    def get(self, oid):
        """
        Get a single record

        :param oid: Object ID such as domain name or zone name
        :param kwargs: Extra options to send to the server
        :return: dictionary of record
        """
        return self.http_get(oid=oid)


class List:

    def list(self):
        """
        Get list of all record.

        :return: list of record dictionaries
        """
        return self.http_get()


class Update:

    def update(self, oid, data):
        """
        Update a single record

        :param data: Data to send to the server to create the resource
        :param oid: Object ID
        :return: dictionary of record
        """
        return self.http_put(oid=oid, post_data=data)


class Delete:

    def delete(self, oid):
        """
        Delete the record

        :param oid: Object ID such as domain name or zone name
        """
        return self.http_delete(oid=oid)


class CRUD(Base, Create, Get, List, Update, Delete):
    pass
