import json
from urllib.parse import urlencode
from lcrequests import Request, Response
from lcrequests.tokens import PasetoToken

from pylogcabin.exceptions import (
    ConsoleAPIError,
    ConsoleAPIAuthenticationError,
    ConsoleAPIAuthorizationError,
    ConsoleAPIClientError,
    ConsoleAPIServerError,
    ConsoleAPINotFoundError,
)
from .tokens import V4PasetoToken
from .typehints import HTTP_VERBS


class PortalAPI:
    """
    Portal API Class.

    Usage:

    lc = PortalAPI(server='https://<url>', token='<token>')
    lc.load(app='accounts', model='company')

    results = lc.get()
    print(results.json())
    """

    API_NAME: str
    SERVER: str
    TOKEN_CLASS: V4PasetoToken

    def __init__(
        self,
        server: str = '',
        token: str = '',
        timeout: int = 30,
        signed_token_class: PasetoToken | None = None,
        **kwargs,
    ):
        """
        :param server: Base URI for Console web interface
        :param token: Auth Token string for Token-based Authorization
        :param timeout: optional connect and read timeout in seconds
        :param signed_token_class: Token object for authenticating
            with signed Paseto Tokens.
        """
        server = server or self.SERVER
        if server.startswith('http'):
            self.server = server
        else:
            self.server = f'https://{server}'
        self.timeout = timeout

        self.headers = {
            'Content-Type': 'application/json; charset=utf-8',
            'User-Agent': f'{self.API_NAME}/1.0',
            'Cache-Control': 'no-cache',
        }
        if 'headers' in kwargs:
            self.headers.update(kwargs['headers'])

        self.base_url = f'{self.server}/api'
        self.url = self.base_url
        self.signed_token_class = signed_token_class

        request_kwargs = {
            'url': self.url,
            'headers': self.headers,
        }

        if signed_token_class:
            request_kwargs['signed_token_class'] = signed_token_class
        else:
            request_kwargs['token'] = token
            request_kwargs['authorization_type'] = kwargs.pop('authorization_type', 'ConsoleAuth')

        self.session: Request = Request(**request_kwargs)

    def load(self, app: str, model: str):
        """
        Generate URL for App / Model
        :param app: Console App
        :param model: Database model within the App
        """
        self.url = f'{self.base_url}/{app}/{model}'
        return self.url

    def get_url(self, cid: str = '', subpath: str = '', params: dict | None = None):
        """
        Generate path params, supporting custom API routes, such as
        /ksync/syncbatch/<cid>/run/
        /latchstring/user/<cid>/set_password/

        Also support dictionary of Query Params
        :param cid: Canonical ID of record to retrieve
        :param subpath: URL path to append to current URL.
            Used for calling special API methods.
        :param params: Data to send as query parameters
        """
        url = self.url if self.url.endswith('/') else f'{self.url}/'
        if cid:
            url = f'{url}{cid}/'

        if subpath:
            url = f'{url}{subpath}/'

        if not params:
            return url

        return f'{url}?{urlencode(params)}'

    def http_request(
        self,
        verb: HTTP_VERBS,
        url: str,
        post_data: dict | list[dict] | None = None,
    ) -> Response:
        """Make an HTTP request to the Log Cabin Console.

        :param verb: The HTTP method to call ('get', 'post', 'put', 'delete')
        :param url: Path or full URL to query (/devices, or /apps/<id>)
        :param post_data: Data to send in the body (will be converted to json)

        Returns a request result object.
        """
        self.session.url = url
        response = self.session.request(
            method=verb,
            url=url,
            data=post_data,
        )

        if response.status >= 400:
            try:
                msg = ','.join(json.loads(response.data)['errors'])
            except (ValueError, TypeError, KeyError):
                if response.reason:
                    msg = response.reason
                else:
                    msg = f'{response.status}: {response.text}'

            if response.status == 401:
                raise ConsoleAPIAuthenticationError(msg, response=response)

            if response.status == 403:
                raise ConsoleAPIAuthorizationError(msg, response=response)

            if response.status == 404:
                # Return response for HEAD existence requests checks as 404 will be expected
                if verb == 'HEAD':
                    return response

                raise ConsoleAPINotFoundError(f'{url} not found', response=response)

            if 400 <= response.status < 500:
                raise ConsoleAPIClientError(msg, response=response)

            raise ConsoleAPIServerError(f'{msg} {response.status}', response=response)

        if response.status == 302:
            raise ConsoleAPIError('Unexpected Redirect', response=response)

        return response

    def head(self, cid: str = '', params: dict | None = None, **kwargs):
        """Make a HEAD request to the Log Cabin server.

        :param params: Data to send as query parameters
        :param cid: Canonical ID of record to retrieve
        :type params: dict

        Returns the response received from the server.
        """
        return self.http_request(
            verb='HEAD',
            url=self.get_url(cid=cid, subpath=kwargs.pop('subpath', ''), params=params),
        )

    def get(self, cid: str = '', params: dict | None = None, **kwargs):
        """Make a GET request to the Log Cabin server.

        :param params: Data to send as query parameters
        :param cid: Canonical ID of record to retrieve
        :type params: dict

        Returns the response received from the server.
        """
        return self.http_request(
            verb='GET',
            url=self.get_url(cid=cid, subpath=kwargs.pop('subpath', ''), params=params),
        )

    def list(self, params: dict | None = None, **kwargs):
        """Make a GET request to the Log Cabin Console.
        Supports pagination by fetching "next" URL until
        no more results are available from the server.

        :param params: Data to send as query parameters
        :type params: dict

        Returns a dict of results:
        {
            "count": 50,
            "results": ['list','of','results'],
        }
        """
        results = self.http_request(
            verb='GET',
            url=self.get_url(subpath=kwargs.pop('subpath', ''), params=params),
        ).json()

        while results.get('next', ''):
            # Don't pass params, because they'll be included
            # in the "next" URL key
            res = self.http_request(
                verb='GET',
                url=results['next'],
            ).json()

            results['next'] = res['next']
            results['results'].extend(res['results'])

        return results

    def list_page(self, params: dict | None = None, **kwargs):
        """Make a GET request to the Log Cabin Console,
        retrieving the list of records that match the
        specified pagination values in the `params` dict.

        To retrieve all the records, update the pagination
        params can call this method again until no results
        are returned.

        :param params: Data to send as query parameters
        :type params: dict

        Returns a dict of results:
        {
            "count": 50,
            "next": 'https://server.com/next/url.html?param=1&limit=50&offset=150',
            "previous": 'https://server.com/next/url.html?param=1&limit=50&offset=100',
            "results": ['list','of','results'],
        }
        """
        return self.http_request(
            verb='GET',
            url=self.get_url(subpath=kwargs.pop('subpath', ''), params=params),
        ).json()

    def canonical_ids(self, params: dict | None = None, **kwargs):  # noqa
        """
        Get Canonical IDs of all records matching specified params

        :param params: Data to send as query parameters
        :type params: dict

        Returns a dict of result:
        {
            "count": 5,
            "results": [
                "1e8a152c-2a98-6ea4-89c6-4439c46d4bd4",
                "1e86f1a4-07c8-6c5c-a4de-4439c46d4bd4",
                "1e7666da-53d4-6714-807c-525400925559",
                "1e82c830-5153-6760-b402-4439c46d4bd4",
                "1ea7db3b-793e-6a8e-9ae0-7085c27cfb9b",
            ]
        }
        """
        return self.http_request(
            verb='GET',
            url=self.get_url(subpath='canonical_ids', params=params),
        ).json()

    def count(self, params: dict | None = None, **kwargs):  # noqa
        """
        Get count of all records matching specified params

        :param params: Data to send as query parameters
        :type params: dict

        Returns a dict with count key:
        {
            "count": 50,
        }
        """
        return self.http_request(
            verb='GET',
            url=self.get_url(subpath='count', params=params),
        ).json()

    def patch(
        self,
        cid: str = '',
        params: dict | None = None,
        post_data: dict | None = None,
        **kwargs,
    ):
        """Make a PATCH request to the Log Cabin Console.

        :param cid: Canonical ID of record to retrieve
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)

        Returns the response received from the server.
        """
        return self.http_request(
            verb='PATCH',
            url=self.get_url(cid=cid, subpath=kwargs.pop('subpath', ''), params=params),
            post_data=post_data,
        )

    def post(
        self,
        cid: str = '',
        params: dict | None = None,
        post_data: dict | None = None,
        **kwargs,
    ):
        """Make a POST request to the Log Cabin Console.

        :param cid: Canonical ID of record to retrieve
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)

        Returns the response received from the server.
        """
        return self.http_request(
            verb='POST',
            url=self.get_url(cid=cid, subpath=kwargs.pop('subpath', ''), params=params),
            post_data=post_data,
        )

    def put(
        self,
        cid: str = '',
        params: dict | None = None,
        post_data: dict | None = None,
        **kwargs,
    ):
        """Make a PUT request to the Log Cabin Console.

        :param cid: Canonical ID of record to retrieve
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)

        Returns the response received from the server.
        """
        return self.http_request(
            verb='PUT',
            url=self.get_url(cid=cid, subpath=kwargs.pop('subpath', ''), params=params),
            post_data=post_data,
        )

    def bulk(
        self, params: dict | None = None, post_data: dict | None = None, method='PUT', **kwargs
    ):
        """Make a request with a list of records to the Log Cabin Console.
        If a dict of post_data is passed in, it'll be converted to a list.
        Verb can be specified with "verb" kwarg. Default verb is PUT.

        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)
        :param method: HTTP method to use in making the request.

        Returns the response received from the server.
        """
        if not post_data:
            raise ValueError('Provide data records to bulk update!')

        return self.http_request(
            verb=method,
            url=self.get_url(cid='', subpath=kwargs.pop('subpath', ''), params=params),
            post_data=[post_data] if isinstance(post_data, dict) else post_data,
        )

    def delete(self, cid: str):
        """Make a DELETE request to the Log Cabin Console.

        :param cid: Canonical ID of record to retrieve
        """
        return self.http_request(
            verb='DELETE',
            url=self.get_url(cid=cid),
        )

    def __str__(self):
        return f'{self.API_NAME} - Server: {self.url}'

    def __repr__(self):
        klass = self.__class__.__name__
        return f'{klass}(server={self.server}, token=<token>, timeout={self.timeout})'


class AspenAPI(PortalAPI):
    """
    Aspen CA Server API wrapper
    """

    API_NAME = 'Aspen CA Server API'
    SERVER = 'https://aspen.compassfoundation.io'


class BeaconAPI(PortalAPI):
    """
    Beacon MDM API wrapper
    """

    API_NAME = 'Beacon MDM API'
    SERVER = 'https://beacon.orbitmobile.io'


class ClavisAuthAPI(PortalAPI):
    """
    Clavis Auth API wrapper
    """

    API_NAME = 'Clavis Auth API'
    SERVER = 'https://clavis.compassfoundation.io'


class DrawBridgeAPI(PortalAPI):
    """
    DrawBridge API wrapper
    """

    API_NAME = 'DrawBridge Portal API'
    SERVER = 'https://127.0.0.1:1525'


LogCabinAPI = DrawBridgeAPI


class VisionMarketAPI(PortalAPI):
    """
    Vision Market API wrapper
    """

    API_NAME = 'Vision Market API'
    SERVER = 'https://market.orbitmobile.io'


__all__ = (
    'AspenAPI',
    'BeaconAPI',
    'ClavisAuthAPI',
    'DrawBridgeAPI',
    'VisionMarketAPI',
    'PortalAPI',
    'LogCabinAPI',
)
