from urllib.parse import urlencode
from lcrequests import Request, Response
from pylogcabin.exceptions import (
    ConsoleAPIError,
    ConsoleAPIAuthenticationError,
    ConsoleAPIAuthorizationError,
    ConsoleAPIClientError,
    ConsoleAPIServerError,
    ConsoleAPINotFoundError,
)
from typing import Optional, TYPE_CHECKING
if TYPE_CHECKING:
    from lcrequests.tokens import PasetoToken
    from lcrequests.typehints import Subject
from .tokens import V4PasetoToken, UmbrellaTokenCentral
from .typehints import DELETE, GET, HEAD, PATCH, POST, PUT


class UmbrellaAPI:
    """
    Umbrella API Class.

    Usage:

    api = UmbrellaAPI(signed_token_class=<signed_token_class>)
    api.load(app='beacon', path='agents')

    results = api.get()
    print(results.json())
    """
    API_NAME: str
    SERVER: str
    TOKEN_CLASS: V4PasetoToken

    def __init__(
        self,
        signed_token_class: 'PasetoToken',
        timeout: int = 30,
        headers: Optional[dict] = None,
    ):
        """
        :param signed_token_class: Token object for authenticating
            with signed Paseto Tokens.
        """
        self.timeout = timeout

        self.headers = {
            'User-Agent': f'{self.API_NAME}/1.0',
            'Cache-Control': 'no-cache',
        }
        if headers:
            self.headers.update(headers)

        self.base_url = f'{self.SERVER}/umbrella/v1'
        self.url = self.base_url

        request_kwargs = {
            'url': self.url,
            'headers': self.headers,
            'signed_token_class': signed_token_class,
        }

        self.session: Request = Request(**request_kwargs)

    def load(self, subject: 'Subject', path: str = '') -> str:
        """
        Generate URL for specified subject and path
        """
        subject_path = f'{subject}/{path}' if path else subject
        self.url = f'{self.base_url}/{subject_path}'
        return self.url

    def get_url(self, cid: str = '', params: dict = None) -> str:
        """
        Generate path params, supporting custom API routes, such as
        /ksync/syncbatch/<cid>/run/
        /latchstring/user/<cid>/set_password/

        Also support dictionary of Query Params
        :param cid: Canonical ID of record to retrieve
        :param params: Data to send as query parameters
        """
        url = f'{self.url}/'
        if cid:
            url = f'{url}{cid}/'

        if not params:
            return url

        return f'{url}?{urlencode(params)}'

    def http_request(self, verb: str, url: str, post_data: dict = None) -> Response:
        """Make an HTTP request to the Umbrella Console.

        :param verb: The HTTP method to call ('get', 'post', 'put', 'delete')
        :param url: Path or full URL to query (/devices, or /apps/<id>)
        :param post_data: Data to send in the body (will be converted to json)

        Returns a request result object.
        """
        if verb in (PATCH, POST, PUT):
            headers = {'Content-Type': 'application/x-www-form-urlencoded'}
        else:
            headers = {}

        self.session.url = url
        response = self.session.request(
            method=verb,
            url=url,
            data=post_data,
            headers=headers,
        )

        if response.status >= 400:
            try:
                msg = ','.join(response.json()['errors'])
            except (ValueError, TypeError, KeyError):
                if response.reason:
                    msg = response.reason
                else:
                    msg = f'{response.status}: {response.text}'

            if response.status == 401:
                raise ConsoleAPIAuthenticationError(msg, response=response)

            if response.status == 403:
                raise ConsoleAPIAuthorizationError(msg, response=response)

            if response.status == 404:
                # Return response for HEAD existence requests checks as 404 will be expected
                if verb == HEAD:
                    return response

                raise ConsoleAPINotFoundError(f'{url} not found', response=response)

            if 400 <= response.status < 500:
                raise ConsoleAPIClientError(msg, response=response)

            raise ConsoleAPIServerError(f'{msg} {response.status}', response=response)

        if response.status == 302:
            raise ConsoleAPIError('Unexpected Redirect', response=response)

        return response

    def head(self, cid: str = '', params: dict = None, **kwargs):
        """Make a HEAD request to the Umbrella server.

        :param params: Data to send as query parameters
        :param cid: Canonical ID of record to retrieve
        :type params: dict

        Returns the response received from the server.
        """
        return self.http_request(
            verb=HEAD,
            url=self.get_url(cid=cid, params=params),
        )

    def get(self, cid: str = '', params: dict = None, **kwargs):
        """Make a GET request to the Umbrella server.

        :param params: Data to send as query parameters
        :param cid: Canonical ID of record to retrieve
        :type params: dict

        Returns the response received from the server.
        """
        return self.http_request(
            verb=GET,
            url=self.get_url(cid=cid, params=params),
        )

    def post(self, cid: str = '', params: dict = None, post_data: dict = None, **kwargs):
        """Make a POST request to the Umbrella Console.

        :param cid: Canonical ID of record to retrieve
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)

        Returns the response received from the server.
        """
        return self.http_request(
            verb=POST,
            url=self.get_url(cid=cid, params=params),
            post_data=post_data,
        )

    def delete(self, cid: str):
        """Make a DELETE request to the Umbrella Console.

        :param cid: Canonical ID of record to retrieve
        """
        return self.http_request(
            verb=DELETE,
            url=self.get_url(cid=cid),
        )

    def __str__(self):
        return f'{self.API_NAME} - Server: {self.url}'

    def __repr__(self):
        klass = self.__class__.__name__
        return f'{klass}(server={self.SERVER}, timeout={self.timeout})'


class CompassUmbrellaAPI(UmbrellaAPI):
    API_NAME = 'Compass Umbrella API'
    TOKEN_CLASS = UmbrellaTokenCentral
    SERVER = 'https://erp.compassfoundation.io'


class CTGroupUmbrellaAPI(UmbrellaAPI):
    API_NAME = 'CT Group Umbrella API'
    TOKEN_CLASS = UmbrellaTokenCentral
    SERVER = 'https://erp.ctgroup.tech'


class MWPOLUmbrellaAPI(UmbrellaAPI):
    API_NAME = 'MWPOL Umbrella API'
    TOKEN_CLASS = UmbrellaTokenCentral
    SERVER = 'https://erp.mwpol.net'


__all__ = (
    'UmbrellaAPI',
    'CompassUmbrellaAPI',
    'CTGroupUmbrellaAPI',
    'MWPOLUmbrellaAPI',
)
