#!/usr/bin/python3

import io
import re
import os
from os.path import dirname, join

from setuptools import setup


def read(*names, **kwargs):
    return io.open(
        join(dirname(__file__), *names),
        encoding=kwargs.get('encoding', 'utf8'),
    ).read()


def get_packages(package):
    """
    Return root package and all sub-packages.
    """
    return [
        dir_path for dir_path, dir_names, filenames in os.walk(package)
        if os.path.exists(os.path.join(dir_path, '__init__.py'))
    ]


def get_package_data(package):
    """
    Return all files under the root package, that are not in a
    package themselves.
    """
    walk = [(dir_path.replace(package + os.sep, '', 1), filenames)
            for dir_path, dir_names, filenames in os.walk(package)
            if not os.path.exists(os.path.join(dir_path, '__init__.py'))]

    file_paths = []
    for base, filenames in walk:
        file_paths.extend([os.path.join(base, filename) for filename in filenames])
    return {package: file_paths}


def get_version():
    with open('pylogcabin/__init__.py', 'r') as f:
        line = f.readline()
        match = re.match(r'__version__ = \'([\d\.]+)\'', line)

        if not match:
            raise ImportError("Can't read the version of pylogcabin")

        version = match.group(1)
        return version


setup(
    name='pylogcabin',
    version=get_version(),
    license='SAP',
    description='Aspen,  Clavis, Log Cabin & Vision Market Console API Clients',
    long_description='%s\n%s' % (
        re.compile('^.. start-badges.*^.. end-badges', re.M | re.S).sub('', read('README.rst')),
        re.sub(':[a-z]+:`~?(.*?)`', r'``\1``', read('CHANGELOG.rst'))
    ),
    author='Dave Burkholder',
    author_email='dave@compassfoundation.io',
    url='https://code.compassfoundation.io/dave/pylogcabin',
    packages=get_packages('pylogcabin'),
    package_data=get_package_data('pylogcabin'),
    zip_safe=True,
    classifiers=[
        'Intended Audience :: Developers',
        'License :: SAP',
        'Operating System :: Unix',
        'Operating System :: POSIX',
        'Programming Language :: Python :: 3.7',
        'Programming Language :: Python :: Implementation :: CPython',
        'Programming Language :: Python :: Implementation :: PyPy',
        'Topic :: Utilities',
    ],
    keywords=[
        'Aspen',
        'DrawBridge',
        'Log Cabin',
        'Clavis',
        'Vision Market',
    ],
    install_requires=[
        'lcrequests',
    ],
)
