import time
from urllib.parse import urlencode

from lcrequests import Request

from . import records
from .exceptions import *

DELETE = 'DELETE'
GET = 'GET'
POST = 'POST'
PUT = 'PUT'
PATCH = 'PATCH'


class Jamf:
    """
    Jamf MDM API Class.
    Represent a connection to a Jamf MDM server.
    """

    def __init__(self, username, password, server_url):
        """
        :param server_url: Base URL for Jamf server. Example: https://jamfserver.example.com:8443
        """
        self.username = username
        self.password = password
        self.url = server_url

        self._token = ''
        self._token_expires = 0.0

        self.apps = records.Apps(self)
        self.buildings = records.Buildings(self)
        self.devices = records.Devices(self)
        self.profiles = records.Profiles(self)
        self.device_groups = records.DeviceGroups(self)
        self.users = records.Users(self)

    def set_token(self):
        """
        Set token and token expiration
        """
        r = self.http_request(POST, 'api/auth/tokens', headers={},
                              username=self.username, password=self.password)
        self._token = r.json()['token']
        self._token_expires = float(r.json()['expires']) / 1000

    @property
    def token(self):
        if not self._token:
            self.set_token()

        elif self._token_expires < time.time():
            self.set_token()
        return self._token

    def get_headers(self, uapi=False):
        """
        Returns headers for request with current auth token
        """
        return {
            'Accept': 'application/json',
            'Content-Type': uapi and 'application/json' or 'text/xml',
            'Authorization': f'Bearer {self.token}',
        }

    @staticmethod
    def _get_querystring(params=None):
        """
        Return dictionary data encoded as URL params
        """
        return params and f'?{urlencode(params)}' or ''

    def build_url(self, path, params=None):
        """
        Generate URL path, supporting custom API routes.

        :param path: Path to resource (/devices, or /apps)
        :param params: Data to send as query parameters
        :type path: str
        :type params: dict
        """
        return f'{self.url}/{path}{self._get_querystring(params)}'

    def http_list(self, path, params=None, **kwargs):
        """Make a GET request to the Jamf server.

        :param path: Path to resource (/devices, or /apps)
        :param params: Data to send as query parameters
        :type path: str
        :type params: dict

        Returns the parsed json returned by the server.
        """
        return self.http_request(GET, path, params=params or {}, **kwargs)

    def http_get(self, path, params=None, **kwargs):
        """Make a GET request to the Jamf server.

        :param path: Path to resource (/devices/<udid>, or /apps/<id>)
        :param params: Data to send as query parameters
        :type path: str
        :type params: dict

        Returns the parsed json returned by the server.
        """
        return self.http_request(GET, path, params=params or {}, **kwargs)

    def http_post(self, path, params=None, post_data=None, **kwargs):
        """Make a POST request to the Jamf server.

        :param path: Path to resource (/devices/<udid>, or /apps/<id>)
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)
        :type path: str
        :type params: dict
        :type post_data: dict

        Returns the parsed json returned by the server.
        """
        return self.http_request(POST, path, params=params or {},
                                 post_data=post_data or {}, **kwargs)

    def http_put(self, path, params=None, post_data=None, **kwargs):
        """Make a PUT request to the Jamf server.

        :param path: Path to resource (/devices/<udid>, or /apps/<id>)
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)
        :type path: str
        :type params: dict
        :type post_data: dict

        Returns the parsed json returned by the server.
        """
        return self.http_request(PUT, path, params=params or {},
                                 post_data=post_data or {}, **kwargs)

    def http_patch(self, path, params=None, post_data=None, **kwargs):
        """Make a PATCH request to the Jamf server.

        :param path: Path to resource (/devices/<udid>, or /apps/<id>)
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)
        :type path: str
        :type params: dict
        :type post_data: dict

        Returns the parsed json returned by the server.
        """
        return self.http_request(PATCH, path, params=params or {},
                                 post_data=post_data or {}, **kwargs)

    def http_delete(self, path, **kwargs):
        """Make a DELETE request to the Jamf server.

        :param path: Path to resource (/devices/<udid>, or /apps/<id>)
        :type path: str
        """
        return self.http_request(DELETE, path, **kwargs)

    def http_request(
            self, verb, path, headers=None, params=None, post_data=None, uapi=False, **kwargs
    ):
        """Make an HTTP request to the Jamf server.

        :param verb: The HTTP method to call ('get', 'post', 'put', 'delete')
        :param path: Path or full URL to query (/devices, or /apps/<id>)
        :param headers: HTTP headers.  self.get_headers() will be called if
        not specified
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)
        :param uapi: Whether to use Pro API or not
        :type verb: str
        :type path: str
        :type params: dict
        :type post_data: dict
        :type uapi: bool

        Returns a request result object.
        """
        url = self.build_url(path, params)
        headers = headers is None and self.get_headers(uapi=uapi) or headers
        response = Request(
            url=url,
            headers=headers,
            username=kwargs.get('username', None),
            password=kwargs.get('password', None),
        ).request(
            method=verb,
            url=url,
            data=post_data,
        )

        if not response.ok:
            try:
                msg = ','.join(response.json()['errors'])
            except (ValueError, TypeError, KeyError):
                if response.reason:
                    msg = response.reason
                else:
                    msg = f'{response.status_code}: {response.text}'

            if response.status_code == 401:
                raise JamfAuthenticationError(msg, response=response)

            if response.status_code == 403:
                raise JamfAuthenticationError(msg, response=response)

            if 400 <= response.status_code < 500:
                raise JamfClientError(msg, response=response)

            raise JamfServerError(
                f'{msg} {response.status_code}', response=response
            )

        if response.status_code == 302:
            raise JamfError('Unexpected Redirect', response=response)

        return response

    def __str__(self):
        return f'Jamf MDM API - Server: {self.url}'

    def __repr__(self):
        return f'Jamf(server={self.url}, auth=<auth>)'


__all__ = [
    'Jamf',
]
