import json
from lcrequests import Request, Response
from typing import Literal, List

from pydrawbridgeos.exceptions import (
    ConsoleAPIError,
    ConsoleAPIAuthenticationError,
    ConsoleAPIAuthorizationError,
    ConsoleAPIClientError,
    ConsoleAPIServerError,
    ConsoleAPINotFoundError,
)

from vyos.configquery import ConfigTreeQuery

ApiPathChoices = (
    ('config_file', 'config-file'),
    ('configure', 'configure'),
    ('generate', 'generate'),
    ('image', 'image'),
    ('reset', 'reset'),
    ('retrieve', 'retrieve'),
    ('show', 'show'),
)
API_PATHS = Literal[tuple(p[0] for p in ApiPathChoices)]


class DrawBridgeApi:
    """
    DrawBridge OS API Class.

    Usage:

    db = DrawBridgeApi(server='https://<url>', key='<key>')

    resp = db.retrieve('showConfig', [])
    print(resp.json())
    """
    API_NAME = 'DrawBridge OS API'
    SERVER = 'http://127.0.0.1:1515'

    def __init__(
        self,
        server: str = '',
        key: str = '',
        **kwargs,
    ):
        """
        :param server: Base URI for Log Cabin web interface
        :param key: API Key for Authorization
        """
        self.server = server or self.SERVER
        self.key = key

        self.headers = {
            'User-Agent': f'{self.API_NAME}/1.0',
            'Cache-Control': 'no-cache',
        }
        if 'headers' in kwargs:
            self.headers.update(kwargs['headers'])

        request_kwargs = {
            'url': self.server,
            'headers': self.headers,
        }

        self.session: Request = Request(**request_kwargs)

    def http_request(self, endpoint: API_PATHS, data: dict) -> Response:
        """Make an HTTP request to the Log Cabin Console.

        :param endpoint: One of the available API Endpoints
        :param data: dict that will be submitted to the API

        Returns a request result object.
        """
        url = f'{self.server}/{endpoint}'
        self.session.url = url

        response = self.session.post(
            fields={
                'key': self.key,
                'data': json.dumps(data),
            },
        )

        if not response.ok:
            try:
                msg = u','.join(response.json()['errors'])
            except (ValueError, TypeError, KeyError):
                if response.reason:
                    msg = response.reason
                else:
                    msg = f'{response.status_code}: {response.text}'

            if response.status_code == 401:
                raise ConsoleAPIAuthenticationError(msg, response=response)

            if response.status_code == 403:
                raise ConsoleAPIAuthorizationError(msg, response=response)

            if response.status_code == 404:
                raise ConsoleAPINotFoundError(f'{url} not found', response=response)

            if 400 <= response.status_code < 500:
                raise ConsoleAPIClientError(msg, response=response)

            raise ConsoleAPIServerError(f'{msg} {response.status_code}', response=response)

        if response.status_code == 302:
            raise ConsoleAPIError('Unexpected Redirect', response=response)

        return response

    def retrieve(self, op: Literal['showConfig', 'returnValues'], path: List[str]) -> Response:
        """
        With the retrieve endpoint you get parts or the whole configuration.

        To get the entire configuration, pass an empty list to the path field.

        :param op: Operation to perform
        :param path: List of strings, such as ["system", "syslog"]
        """
        return self.http_request('retrieve', {'op': op, 'path': path})

    def reset(self, path: List[str]) -> Response:
        """
        The `reset` endpoint run a `reset` command.

        :param path: List of strings, such as ["ip", "bgp", "192.0.2.11"]
        """
        return self.http_request('reset', {'op': 'reset', 'path': path})

    def image(self, op: Literal['add', 'delete'], name: str) -> Response:
        """
        To add or delete an image, use the `/image` endpoint.

        :param op: Operation to perform
        :param name: Name of image
        """
        return self.http_request('image', {'op': op, 'name': name})

    def show(self, path: List[str]) -> Response:
        """
        The `/show` endpoint is to show everything in the operational mode.

        For example, show which images are installed.

        :param path: List of strings, such as ["system", "image"]
        """
        return self.http_request('show', {'op': 'show', 'path': path})

    def generate(self, path: List[str]) -> Response:
        """
        The `generate` endpoint run a `generate` command.

        :param path: List of strings, such as ["wireguard", "default-keypair"]
        """
        return self.http_request('generate', {'op': 'generate', 'path': path})

    def configure(self, op: Literal['set', 'delete', 'comment'], path: List[str]) -> Response:
        """
        The `generate` endpoint run a generate command.

        :param op: Operation to perform
        :param path: List of strings, such as ["wireguard", "default-keypair"]
        """
        return self.http_request('configure', {'op': op, 'path': path})

    def config_file(self, op: Literal['load', 'save'], location: str = '') -> Response:
        """
        The endpoint `/config-file` is to save or load a configuration.

        Save a running configuration to the startup configuration.
        When you don’t specify the file when saving, it saves to /config/config.boot.

        :param op: Operation to perform
        :param location: Location to save config file
        """
        data = {'op': op}
        if location:
            data['file'] = location
        return self.http_request('config-file', data)

    def __str__(self):
        return f'{self.API_NAME} - Server: {self.server}'

    def __repr__(self):
        klass = self.__class__.__name__
        return f'{klass}(server={self.server}, key=<key>)'


class LocalDrawBridgeApi(DrawBridgeApi):

    def __init__(self):
        super().__init__(key=self.lookup_local_key())

    def lookup_local_key(self) -> str:
        path = ['service', 'https', 'api', 'keys', 'id', 'DEFAULT-DRAWBRIDGEOS-API-ID']
        cfg = ConfigTreeQuery()
        if cfg.exists(path):
            return cfg.value(path + ['key'])
        else:
            raise Exception('The default API key is not set.')


__all__ = (
    'ApiPathChoices',
    'DrawBridgeApi',
    'LocalDrawBridgeApi'
)
