from typing import TYPE_CHECKING
from .mixins import (
    Create,
    Update,
    CRUD,
    Read,
)
if TYPE_CHECKING:
    from .beacon import Beacon


class BeaconRecord:
    """Base class for CRUD operations on Beacon Records.

    Subclasses class must define `path`.

    `path`: Base URL path on which requests will be sent (e.g. '/api/device')
    """
    path = None

    def __init__(self, beacon: 'Beacon'):
        """MDMObject manager constructor.

        :param beacon: `beacon.Beacon` connection to make requests
        """
        self.beacon: 'Beacon' = beacon

    def change_cid(self, old: str, new: str, params: dict = None, **kwargs):
        """
        Change the CID of a record in Beacon

        :param old: old CID
        :param new: new CID
        :param params: Data to send as query parameters
        :param kwargs: Extra options to send to the server
        :return: dictionary of record
        """
        path = kwargs.pop('path', None) or f'{self.path}/change-cid'
        params = params or {}
        params['oldCid'] = old
        params['newCid'] = new
        return self.beacon.http_post(path, params=params, **kwargs)


class App(CRUD, BeaconRecord):
    path = 'api/app'


class Company(CRUD, BeaconRecord):
    path = 'api/company'


class Device(CRUD, BeaconRecord):
    path = 'api/device'

    def new_license(self, params: dict = None, **kwargs):
        """
        Get a new MDM license code

        :param params: Data to send as query parameters
        :param kwargs: Extra options to send to the server
        :return: new license code
        """
        path = kwargs.pop('path', None) or 'api/device/new-license'
        return self.beacon.http_get(path, params=params, **kwargs)

    def poke(self, oid, params: dict = None, **kwargs):
        """
        Poke a device to Refresh Inventory and Start all created commands

        :param oid: ID, such as serial number, license code, or numerical ID
        :param params: Data to send as query parameters
        :param kwargs: Extra options to send to the server
        :return: json of response, normally empty dict
        """
        path = kwargs.pop('path', None) or f'api/device/poke/{oid}'
        return self.beacon.http_post(path, params=params, **kwargs)


class DeviceCommand(Create, Read, Update, BeaconRecord):
    path = 'api/device-command'


class Market(CRUD, BeaconRecord):
    path = 'api/market'


class Permission(CRUD, BeaconRecord):
    path = 'api/permission'


class Policy(CRUD, BeaconRecord):
    path = 'api/policy'


class Profile(CRUD, BeaconRecord):
    path = 'api/profile'


class ProfileBoolSetting(CRUD, BeaconRecord):
    path = 'api/profile/bool-setting'


class ProfileDeviceJoin(CRUD, BeaconRecord):
    path = 'api/profile-device-join'


class ProfileStringSetting(CRUD, BeaconRecord):
    path = 'api/profile/string-setting'


class Setting(CRUD, BeaconRecord):
    path = 'api/setting'


class User(CRUD, BeaconRecord):
    path = 'api/user'
