from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from .records import BeaconRecord
    MixinBase = BeaconRecord
else:
    MixinBase = object


class Create(MixinBase):

    def create(self, data, params=None, **kwargs):
        """
        Create a new record.

        :param data: Data to send to the server to create the resource
        :param params: Data to send as query parameters
        :param kwargs: Extra options to send to the server
        :return: dictionary of record
        """
        path = kwargs.pop('path', None) or self.path
        return self.beacon.http_post(path, post_data=data, params=params, **kwargs)


class Get(MixinBase):

    def get(self, oid, params=None, **kwargs):
        """
        Get a single record

        :param oid: ID, such as serial number, udid or numerical ID
        :param params: Data to send as query parameters
        :param kwargs: Extra options to send to the server
        :return: dictionary of record
        """
        path = kwargs.pop('path', None) or f'{self.path}/{oid}'
        return self.beacon.http_get(path, params=params, **kwargs)


class List(MixinBase):

    def list(self, params=None, **kwargs):
        """
        Get list of all record.

        :param params: Data to send as query parameters
        :param kwargs: Extra options to send to the server
        :return: list of record dictionaries
        """
        path = kwargs.pop('path', None) or self.path
        return self.beacon.http_get(path, params=params, **kwargs)


class Update(MixinBase):

    def update(self, oid, data, params=None, **kwargs):
        """
        Update a single record

        :param data: Data to send to the server to create the resource
        :param oid: ID, such as serial number, udid or numerical ID
        :param params: Data to send as query parameters
        :param kwargs: Extra options to send to the server
        :return: dictionary of record
        """
        path = kwargs.pop('path', None) or f'{self.path}/{oid}'
        return self.beacon.http_post(path, params=params, post_data=data, **kwargs)


class Delete(MixinBase):

    def delete(self, oid, **kwargs):
        """
        Delete a record

        :param oid: ID, such as serial number, udid or numerical ID
        """
        path = f'{self.path}/{oid}'
        return self.beacon.http_delete(path, **kwargs)


class CRUD(Create, Get, List, Update, Delete):
    pass


class Read(Get, List):
    pass
