from lcrequests import Request, Response
import time
from urllib.parse import urlencode

from . import records
from .exceptions import *

DELETE = 'DELETE'
GET = 'GET'
POST = 'POST'
PUT = 'PUT'
PATCH = 'PATCH'


class Beacon:
    """
    Beacon MDM API Class.
    Represent a connection to a Beacon MDM server.

    Usage:

    beacon = Beacon(server='https://<url>', token='<token>')

    response = beacon.devices.list()
    """

    def __init__(self, username, password, server_url):
        """
        :param server_url: Base URL for Beacon server. Example: https://beacon.example.com:5001
        """
        self.username = username
        self.password = password
        self.server_url = server_url.strip().strip('/')
        self.url = self.server_url
        self._token = ''
        self._token_expires = 0
        self._session = None

        self.apps = records.App(self)
        self.companies = records.Company(self)
        self.devices = records.Device(self)
        self.device_commands = records.DeviceCommand(self)
        self.markets = records.Market(self)
        self.permissions = records.Permission(self)
        self.policies = records.Policy(self)
        self.profiles = records.Profile(self)
        self.profile_bool_settings = records.ProfileBoolSetting(self)
        self.profile_device_joins = records.ProfileDeviceJoin(self)
        self.profile_string_settings = records.ProfileStringSetting(self)
        self.settings = records.Setting(self)
        self.users = records.User(self)

    def set_token(self):
        """
        Set token and token expiration
        """
        headers = {
            'Accept': 'application/json',
            'Content-Type': 'application/json',
        }
        data = {
            'email': self.username,
            'password': self.password,
        }
        url = f'{self.url}/api/user/login'
        response = Request(url=url, headers=headers).request(method=POST, url=url, data=data).json()

        self._token = response.get('token')
        self._token_expires = response.get('tokenExpiration')

    @property
    def session(self):
        if not self._session:
            headers = {
                'Accept': 'application/json',
                'Content-Type': 'application/json',
                'Authorization': f'Bearer {self.token}'
            }
            self._session = Request(url=self.url, headers=headers)
        return self._session

    @property
    def token(self):
        if not self._token:
            self.set_token()

        if self._token_expires < int(time.time()):
            self.set_token()
        return self._token

    @staticmethod
    def _get_querystring(params=None) -> str:
        """
        Return dictionary data encoded as URL params
        """
        return params and f'?{urlencode(params)}' or ''

    def build_url(self, path: str, params: dict = None) -> str:
        """
        Generate URL path, supporting custom API routes.

        :param path: Path to resource (/devices, or /apps)
        :param params: Data to send as query parameters
        """
        if path.startswith('https://'):
            url = path
        elif path.startswith('/'):
            url = f'{self.server_url}{path}'
        else:
            url = f'{self.server_url}/{path}'
        return f'{url}{self._get_querystring(params)}'

    def http_list(self, path: str, params: dict = None, **kwargs):
        """Make a GET request to the Beacon server.

        :param path: Path to resource (/devices, or /apps)
        :param params: Data to send as query parameters

        Returns the parsed json returned by the server.
        """
        return self.http_request(GET, path, params=params or {}, **kwargs)

    def http_get(self, path: str, params: dict = None, **kwargs):
        """Make a GET request to the Beacon server.

        :param path: Path to resource (/devices/<udid>, or /apps/<id>)
        :param params: Data to send as query parameters

        Returns the parsed json returned by the server.
        """
        return self.http_request(GET, path, params=params or {}, **kwargs)

    def http_post(self, path: str, params: dict = None, post_data: dict = None, **kwargs):
        """Make a POST request to the Beacon server.

        :param path: Path to resource (/devices/<udid>, or /apps/<id>)
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)

        Returns the parsed json returned by the server.
        """
        return self.http_request(
            POST, path, params=params or {}, post_data=post_data or {}, **kwargs
        )

    def http_put(self, path: str, params: dict = None, post_data: dict = None, **kwargs):
        """Make a PUT request to the Beacon server.

        :param path: Path to resource (/devices/<udid>, or /apps/<id>)
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)

        Returns the parsed json returned by the server.
        """
        return self.http_request(
            PUT, path, params=params or {}, post_data=post_data or {}, **kwargs
        )

    def http_patch(self, path: str, params: dict = None, post_data: dict = None, **kwargs):
        """Make a PATCH request to the Beacon server.

        :param path: Path to resource (/devices/<udid>, or /apps/<id>)
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)

        Returns the parsed json returned by the server.
        """
        return self.http_request(
            PATCH, path, params=params or {}, post_data=post_data or {}, **kwargs
        )

    def http_delete(self, path: str, **kwargs):
        """Make a DELETE request to the Beacon server.

        :param path: Path to resource (/devices/<udid>, or /apps/<id>)
        :type path: str
        """
        return self.http_request(DELETE, path, **kwargs)

    def http_request(
            self,
            verb: str,
            path: str,
            params: dict = None,
            post_data: dict = None,
            **kwargs,  # noqa
    ) -> Response:
        """Make an HTTP request to the Beacon server.

        :param verb: The HTTP method to call ('get', 'post', 'put', 'delete')
        :param path: Path or full URL to query (/devices, or /apps/<id>)
        :param params: Data to send as query parameters
        :param post_data: Data to send in the body (will be converted to json)

        Returns a request result object.
        """
        url = self.build_url(path, params)
        self.session.url = url
        response = self.session.request(method=verb, url=url, data=post_data)

        if not response.ok:
            try:
                msg = ','.join(response.json()['errors'])
            except (ValueError, TypeError, KeyError):
                if response.reason:
                    msg = f'{response.reason} -- {url}'
                else:
                    msg = f'{response.status_code}: {response.text} -- {url}'

            if response.status_code == 401:
                raise BeaconAuthenticationError(msg, response=response)

            if response.status_code == 403:
                raise BeaconAuthorizationError(msg, response=response)

            if 400 <= response.status_code < 500:
                raise BeaconClientError(msg, response=response)

            raise BeaconServerError(f'{msg} {response.status_code} -- {url}', response=response)

        if response.status_code == 302:
            raise BeaconError('Unexpected Redirect', response=response)

        return response

    def __str__(self):
        return f'Beacon MDM API - Server: {self.url}'

    def __repr__(self):
        return f'Beacon(server={self.url}, token=<token>)'


__all__ = [
    'Beacon',
]
