# pyBeacon

This is a Python library for communicating with the Beacon MDM via its REST API.

All useful endpoints of the API are implemented as objects of a class.
You may call read and write operations on these objects.

The library also has `http_get`, `http_post`, `http_put`, `http_patch`, and `http_delete` functions that
accept arbitrary URLs.

## Install

Install by cloning the repo and run `setup.py`.
```bash
% python3 setup.py install --user
```

## Usage

To initialize do the following.
```python
from pybeacon import Beacon
api = Beacon('https://beacon.example.com:5001', '<token>')
```

Here are some examples of what you can do with the generic built-in functions.
```python
api.http_get(url, params=params)
api.http_post(url, data=data)
api.http_delete(url)
```

Here is a list of available objects representing API endpoints.
```python
api.apps                        # Interface with app records
api.companies                   # Interface with company records
api.devices                     # Interface with device records
api.device_commands             # Create device management commands
api.markets                     # Interface with market records
api.policies                    # Interface with policy records
api.profiles                    # Interface with configuration profiles
api.profile_bool_settings       # Administrate settings on a device
api.profile_string_settings     # Administrate settings on a device
api.settings                    # Administrate settings on a device
api.users                       # Interface with MDM user records
```

You can call the following operations on the objects.
Some objects may not support all CRUD operations, consult the [API docs]() for more details.
```python
create(data)
get(record_id)
list()
update(record_id, data)
delete(record_id)
```

## Examples

```python
# List all Devices
api.devices.list()

# Create a new device
api.devices.create({
    "name": "Tom's 220",
})

# Get device details
api.devices.get(device_id)

# Update a device
api.devices.update(device_id, {
    "name": "Tom's 420",
})

# Delete a device
api.devices.delete(device_id)

# Create a command to send to a device
commands = {    
    'LockDevice': 0,
    'UnmanageDevice': 1,
    'RestartDevice': 2,
    'WipeDevice': 3,
    'ClearPasscode': 7,
}
api.device_commands.create({
    "type": commands['LockDevice'],
    "deviceId": device_id,
})

# View the pending commands for a certain device
api.device_commands.get(device_id)

# Start all pending commands with special poke function
api.devices.poke(device_id)
```
