from base64 import standard_b64decode, standard_b64encode
import configparser
from datetime import datetime, timedelta
import os
from typing import Union
from uuid import UUID

import json
from lcconfig import CareCenterSettingsConfig
from lcrequests import Request
from .utils import datefromiso

RecordId = Union[int, str, UUID]


# ---------------------------------------------------------------------------
class BaseOdoo:
    """
    Base Odoo API Class. Prefer Odoo class and specified
    config file to cache token between instantiations.
    """

    model = ''

    def __init__(self, url, uname, pw, db):
        if url and not url.startswith('http'):
            url = f'https://{url}'
        self.url = url
        self.uname = uname
        self.pw = pw
        self.db = db
        self.token_expires = None

        # # base headers for when authentication not available / required
        self.base_headers = {
            'content-type': 'application/x-www-form-urlencoded',
            'charset': 'utf-8',
        }
        self.headers = {
            'content-type': 'application/x-www-form-urlencoded',
            'charset': 'utf-8',
            'access-token': '',
        }

    def _get_headers(self) -> dict:
        """
        Return containing the request headers
        """
        if self.headers['access-token']:
            return self.headers

        self.authenticate()

        return self.headers

    def authenticate(self):
        """
        Add server authentication token to headers if
        one does not already exist.
        """
        if not self.headers['access-token']:
            self._login()
        elif not self.token_expires or self.token_expires < datetime.now():
            self._login()

    def _login(self):
        headers = self.base_headers.copy()
        headers.update({
            'db': self.db,
            'login': self.uname,
            'password': self.pw,
        })

        req = Request(url=f'{self.url}/api/auth/token', headers=headers)
        resp = req.post()

        if resp.status >= 400:
            self.handle_http_error(resp)

        r_data = resp.json()
        if 'access_token' not in r_data:
            self.handle_http_error(resp)

        self.headers['access-token'] = r_data.get('access_token')
        expires_in = int(r_data.get('expires_in'))
        self.token_expires = datetime.now() + timedelta(seconds=expires_in)

    def load(self, model: str):
        """
        Set the Odoo model env
        """
        self.model = model

    # -------------------------------------------------------------------------------------
    # Odoo API Methods (wrapping HTTP methods)
    # -------------------------------------------------------------------------------------

    def search(self, domain: list, fields: tuple = ()) -> list:
        """
        Search the API database using Odoo's API search
        syntax.  Return a dict with all the values in the
        matched rows unless fields param is specified.

        Sample response:
        [
            {'id': 2895, 'name': 'John Smith'},
            {'id': 3092, 'name': 'Jane Smith'},
        ]
        """
        self.check_load()
        params = {
            'domain': domain,
            'fields': fields,
        }

        return self.get(f'api/{self.model}', params=params)

    def browse(self, record_id: RecordId, fields: tuple = (), lookup_field='id') -> dict:
        """
        Browse the API database using Odoo's API browse
        syntax.  Return a dict with all the values in the
        matched rows unless fields param is specified.
        Return empty dict if record does not exist.

        Sample response:
        {'email': 'john@example.com',
         'id': 55,
         'name': 'John Smith'
        }
        """
        self.check_load()
        fields = list(fields)
        if lookup_field not in fields:
            fields.append(lookup_field)

        params = {
            'domain': [(lookup_field, '=', record_id)],
            'fields': fields,
        }

        return self.get(f'api/{self.model}', params=params)

    def scid(self, cid: UUID, fields: tuple = ()) -> dict:
        """
        Same as `browse` method except lookup field is `cid`.
        """
        return self.browse(record_id=cid, fields=fields, lookup_field='cid')

    def create(self, create_data: dict) -> int:
        """
        Create an item using Odoo's API create syntax.
        Return the ID of the new record.

        Sample response:
        12634
        """
        self.check_load()

        return self.post(f'api/{self.model}', data=create_data)

    def write(self, record_id: RecordId, write_data: dict) -> bool:
        """
        Update an item using the record ID.  Returns
        True if successful
        """
        self.check_load()

        return self.put(f'api/{self.model}/{record_id}', data=write_data)

    def check_load(self):
        """
        Check if model is loaded and raise exception
        if it not
        """
        if not self.model:
            raise Exception('Model not specified')

    def unlink(self, record_id: RecordId) -> bool:
        """
        Delete an item using the record ID.  Return
        True if deleted successfully.
        """
        self.check_load()
        return self.delete(f'api/{self.model}/{record_id}')

    def rpc(self, record_id: RecordId = 0, method: str = '', method_args: dict = None):
        """
        Call a method on a specific model record.
        """
        self.check_load()

        json_method_args = {
            'params': method_args,
        }
        return self.patch(f'api/{self.model}/{record_id}/{method}', data=json_method_args)

    # -------------------------------------------------------------------------------------
    # HTTP Methods (accessed mainly through Odoo wrapper methods)
    # -------------------------------------------------------------------------------------

    def get(self, url_path: str, params: dict = None):
        headers = self._get_headers().copy()
        params = params or {}
        if params:
            headers.update({
                'domain': json.dumps(params.get('domain', {})),
                'fields': json.dumps(params.get('fields', [])),
                'limit': params.get('limit', 0),
                'offset': params.get('offset', 0),
                'order': json.dumps(params.get('order', None)),
            })

        req = Request(url=f'{self.url}/{url_path}', headers=headers)
        resp = req.get()

        if resp.status >= 400:
            return self.handle_response(
                resp,
                self.get,
                url_path,
                params,
            )

        return resp

    def post(self, url_path: str, data: dict = None):
        data = data or {}

        req = Request(url=f'{self.url}/{url_path}', headers=self._get_headers())
        resp = req.post(body=data)

        if resp.status >= 400:
            return self.handle_response(
                resp,
                self.post,
                url_path,
                data,
            )

        return resp

    def delete(self, url_path):
        req = Request(url=f'{self.url}/{url_path}', headers=self._get_headers())
        resp = req.delete()

        if resp.status >= 400:
            return self.handle_response(
                resp,
                self.delete,
                url_path,
            )

        return resp

    def patch(self, url_path: str, data: dict = None):
        data = data or {}
        req = Request(url=f'{self.url}/{url_path}', headers=self._get_headers())
        resp = req.patch(body=data)

        if resp.status >= 400:
            return self.handle_response(
                resp,
                self.patch,
                url_path,
                data,
            )

        return resp

    def put(self, url_path: str, data: dict = None):
        data = data or {}
        req = Request(url=f'{self.url}/{url_path}', headers=self._get_headers())
        resp = req.put(body=data)

        if resp.status >= 400:
            return self.handle_response(
                resp,
                self.put,
                url_path,
                data,
            )

        return resp

    def handle_http_error(self, http_resp):
        print(
            f'Request failed.  Status code: {http_resp.status_code},  '
            f'Server Message: {http_resp.text}'
        )
        raise SystemExit()

    def handle_response(self, response, caller, *args):
        """
        Want to refresh access token if a http 401 code is thrown.
        @param response: Response
        @param caller: Func, Function that called this function so it can be possibly re-ran
        """
        if response.status_code == 401:
            try:
                data = response.json()
                if data['type'] == 'access_token' and all(
                        (word in data['message'] for word in {'expired', 'token', 'invalid'})
                ):
                    self._login()
                    return caller(*args)
            except:
                pass
        return response


# ---------------------------------------------------------------------------
class BaseConfigOdoo(BaseOdoo):
    """
    Odoo API wrapper class with credentials provided via config file.
    Access token is saved to config file to save login time across
    class instantiations.
    """

    def __init__(self, config_file: str):

        self.cfg = self.load_config(config_file)
        if not self.cfg:
            print('Please setup your account credentials!')
            return

        super().__init__(**self.cfg['Connection'])

        try:
            self.headers['access-token'] = self.cfg['Token']['token']
            self.token_expires = self.cfg['Token']['token_expires']
        except KeyError:
            pass

    def _login(self):
        """
        Login and save access token to config file
        """
        super()._login()

        token = self.headers['access-token']
        if token and self.token_expires:
            self.save_token(token)

    def load_config(self, config_file):
        """
        Define on subclasses to customize load behavior.
        """
        raise NotImplementedError

    def save_token(self, token):
        """
        Define on subclasses to customize save behavior.
        """
        raise NotImplementedError


# ---------------------------------------------------------------------------
class ConsoleOdoo(BaseConfigOdoo):
    """
    Uses encrypted ConfigParser from lcconfig for Console deployments.
    """

    def __init__(self):
        super().__init__(config_file='')

    def load_config(self, config_file: str = ''):

        cfg = CareCenterSettingsConfig()
        return {
            'Connection': cfg.as_typed_dict('Connection'),
            'Token': cfg.as_typed_dict('Token'),
        }

    def save_token(self, token):

        token = standard_b64encode(token.encode('utf8'))
        cfg = CareCenterSettingsConfig()
        data = {
            'token': token.decode('utf8'),
            'expires': str(self.token_expires),
        }
        cfg.save_section('Token', data)


# ---------------------------------------------------------------------------
class Odoo(BaseConfigOdoo):
    """
    Uses stock ConfigParser to ease dependencies for desktop users.
    """

    def load_config(self, config_file: str):
        """
        Load configuration file. Override on subclasses to modify functionality.
        This class uses stock ConfigParser to ease dependencies for desktop users.
        """
        if config_file.startswith('~/'):
            self.config_file = os.path.expanduser(f'~/{config_file[2:]}')
        else:
            self.config_file = config_file

        cfg = configparser.ConfigParser()
        cfg.read(self.config_file)

        if not cfg.has_section('Connection'):
            return

        try:
            password = standard_b64decode(cfg['Connection']['password']).decode('utf8')
        except Exception:
            password = cfg['Connection']['password']

        try:
            token_dict = {
                'token': standard_b64decode(cfg['Token']['token']).decode('utf8'),
                'token_expires': datefromiso(cfg['Token']['expires']),
            }
        except KeyError:
            token_dict = {}

        return {
            'Connection': {
                'url': cfg['Connection']['server_url'],
                'uname': cfg['Connection']['email'],
                'db': cfg['Connection']['database'],
                'pw': password,
            },
            'Token': token_dict,
        }

    def save_token(self, token):
        """
        Save token to reuse across multiple class instantiations.
        Override on subclasses to customize save behavior.
        """
        token = standard_b64encode(token.encode('utf8'))

        cfg = configparser.ConfigParser()
        cfg.read(self.config_file)
        if not cfg.has_section('Token'):
            cfg.add_section('Token')

        cfg['Token'] = {
            'token': token.decode('utf8'),
            'expires': str(self.token_expires),
        }

        with open(self.config_file, 'w') as configfile:
            cfg.write(configfile)


__all__ = (
    'BaseOdoo',
    'BaseConfigOdoo',
    'ConsoleOdoo',
    'Odoo',
)
