#!/usr/bin/python3
import distro
import io
import re
import os
from os.path import dirname, join
import sys

from setuptools import setup

v = sys.version_info


def read(*names, **kwargs):
    return io.open(join(dirname(__file__), *names), encoding=kwargs.get('encoding', 'utf8')).read()


def get_data_files():
    IS_DRAWBRIDGE_OS = distro.name().lower() in ('drawbridgeos', 'vyos', 'ubuntu')
    if IS_DRAWBRIDGE_OS:
        return [
            ('/usr/lib/systemd/systemd', [os.path.join('config/systemd', 'logtailer.service')]),
        ]

    return [
        ('/etc/supervisord.d', [os.path.join('config/supervisord', 'logtailer.conf')]),
    ]


def get_version():
    with open('logtailer/__init__.py', 'r') as f:
        line = f.readline()
        match = re.match(r'__version__ = \'([\d\.]+)\'', line)

        if not match:
            raise ImportError("Can't read the version of logtailer")

        version = match.group(1)
        return version


def get_packages(package):
    """
    Return root package and all sub-packages.
    """
    return [
        dirpath
        for dirpath, dirnames, filenames in os.walk(package)
        if os.path.exists(os.path.join(dirpath, '__init__.py'))
    ]


setup(
    name='logtailer',
    version=get_version(),
    license='SAP',
    description='Tail log files on the system and publish to console and websocket server',
    long_description='%s\n%s'
    % (
        re.compile(r'^.. start-badges.*^.. end-badges', re.M | re.S).sub('', read('README.md')),
        re.sub(r':[a-z]+:`~?(.*?)`', r'``\1``', read('CHANGELOG.md')),
    ),
    author='Dave Burkholder',
    author_email='dave@compassfoundation.io',
    url='https://code.compassfoundation.io/dave/logtailer',
    packages=get_packages('logtailer'),
    data_files=get_data_files(),
    zip_safe=True,
    options={
        'bdist_rpm': {
            'post_install': 'post_install.sh',
            'pre_install': 'pre_install.sh',
        },
    },
    classifiers=[
        'Intended Audience :: Developers',
        'License :: SAP',
        'Operating System :: Unix',
        'Operating System :: POSIX',
        'Programming Language :: Python :: 3.11',
        'Programming Language :: Python :: 3.12',
        'Programming Language :: Python :: Implementation :: CPython',
        'Programming Language :: Python :: Implementation :: PyPy',
        'Topic :: Utilities',
    ],
    keywords=[
        'log processing',
        'DrawBridge Portal',
    ],
    install_requires=[
        'distro',
        'lcconfig',
        'lchttp',
        'watchdog',
    ],
    entry_points={
        "console_scripts": [
            "logtailer = logtailer.__main__:main",
        ]
    },
)
