import json
from lcconfig import ConsoleSettingsConfig
import logging.config
import sys
from system_env.environment import setup_portal_environment
from typing import TYPE_CHECKING
import yaml

if TYPE_CHECKING:
    from celery import Celery

PORTAL_ENV = setup_portal_environment()

csc = ConsoleSettingsConfig().load()
console_config = csc.as_typed_dict('BoxData')
IS_DRAWBRIDGE_OS = PORTAL_ENV['IS_DRAWBRIDGE_OS']
PORTAL_CODE_NAME = PORTAL_ENV['PORTAL_CODE_NAME']
PROJECT_ROOT = PORTAL_ENV['PROJECT_ROOT']
REDWOOD_LOG_DIR = '/var/log/redwood'
TMP_DIR = f'/var/tmp/{PORTAL_CODE_NAME}'

sys.path.insert(0, PROJECT_ROOT)

try:
    # logtailer is installed before drawbridge_portal, portal_app
    # won't exist on initial install, so don't crash
    from celery_init import portal_app  # noqa
except (ImportError, ModuleNotFoundError):
    portal_app: 'Celery' = None  # type: ignore[no-redef]

from console_settings import (  # noqa
    DATETIME_FMT,  # noqa
    DNS_LOGLINE_FIELDS,  # noqa
    LC_CAT_PREFIX,  # noqa
    NON_SCORERS,  # noqa
    REDWOOD_LOG_FIELDS,  # noqa
    ReporterTask,  # noqa
    TLS_LOG_FIELDS,  # noqa
)

# Number of log lines to buffer before POSTing to console
BUFFER_LINES = 500

HOSTNAME = console_config.get('hostname')
FILTER_SERVER = console_config.get('filter_server', True)
CLOUD_SERVER = console_config.get('cloud_server', True)
AUTH_TOKEN = console_config.get('auth_token', '')


def load_centrifugo_config():
    try:
        with open('/etc/centrifugo/config.json', 'r') as csf:
            return json.load(csf)
    except FileNotFoundError:
        pass

    with open(f'{PROJECT_ROOT}/privatesettings/centrifugo.yaml', 'r') as csf:
        return yaml.safe_load(csf)


cc = load_centrifugo_config()
WEBSOCKET_PORT = 10100
CENTRIFUGE_SECRET = cc.get('token_hmac_secret_key')
CENTRIFUGE_API_KEY = cc.get('api_key')
CENTRIFUGE_TIMEOUT = 10
CENTRIFUGE_ADDRESS = f'wss://{HOSTNAME}'

LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'filters': {
        'require_debug_false': {
            '()': 'django.utils.log.RequireDebugFalse',
        }
    },
    'formatters': {
        'default_formatter': {
            'format': "[%(asctime)s] %(levelname)s [%(name)s:%(lineno)s] %(message)s",
            'datefmt': DATETIME_FMT,
        },
    },
    'handlers': {
        'file': {
            'level': 'INFO',
            'class': 'logging.FileHandler',
            'filename': f'/var/log/{PORTAL_CODE_NAME}/logtailer.log',
            'formatter': 'default_formatter',
        },
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'default_formatter',
        },
    },
    'loggers': {
        'logtailer': {
            'handlers': ['file'],
            'propagate': True,
            'level': 'INFO',
        },
    },
}

logging.config.dictConfig(LOGGING)
