import logging
import time
from lchttp.json import json_dumps

from .settings import (
    portal_app,
    ReporterTask,  # noqa
    TMP_DIR,
)

logger = logging.getLogger(__name__)


# ----------------------------------------------------------------------
class PersistLogs:
    """
    Handle saving log lines to shared memory, and
    POSTing notifications to relevant Console API endpoint.

    Also ensure that logs are being processed in a timely
    manner. If not, we should sleep rather than process
    new loglines, to let the console catch up.
    """

    console_app: str = ''
    console_model: str = ''
    task_name: str = ''

    def __init__(self) -> None:
        self.queue_path = f'{TMP_DIR}/{self.console_app}/{self.console_model}'

    def __str__(self) -> str:
        return f'{self.__class__.__name__} for {self.console_app} / {self.console_model}'

    def __repr__(self) -> str:
        return f'{self.__class__.__name__}()'

    def save(self, lines: list[dict]):
        """
        Takes log lines from log tailer and buffer
        them and handle post to console API.
        """
        batch_name = f'{str(time.time())}.json'
        if self.add_to_queue(batch_name, batch_data=lines):
            return self.send_task(batch_name)

    def add_to_queue(self, batch_name: str, batch_data: list) -> bool:
        """
        Add batch data to batch_name key, and then save batch_name to disk.
        """
        batch = f'{self.queue_path}/{batch_name}'

        try:
            with open(batch, 'wb') as bq:
                bq.write(json_dumps(batch_data))
            return True

        except Exception as e:
            logger.exception(f'Error saving batch of {len(batch_data)} lines to {batch}: {e}')

        return False

    def send_task(self, batch: str):
        """
        Send task scheduler that a batch was just completed for processing.
        """
        try:
            # Don't include batch name in task args! The Task is responsible
            # to walk the directory and load all unprocessed batches
            resp = portal_app.send_task(self.task_name)
            logger.info(f'Sent batch {batch!r} to task {self.task_name!r}')
            return resp
        except Exception as e:
            logger.exception(f'Unable to send task {self.task_name!r} with args {batch!r}: {e}')
            return


# ----------------------------------------------------------------------
class PersistRedwoodLogs(PersistLogs):
    console_app = 'reporter'
    console_model = 'logline'
    task_name = ReporterTask.PROCESS_REDWOOD_LOG_LINES


# ----------------------------------------------------------------------
class PersistDNSLogs(PersistLogs):
    console_app = 'firewall'
    console_model = 'rpzlog'
    task_name = ReporterTask.PROCESS_DNS_LOG_LINES


__all__ = (
    'PersistLogs',
    'PersistRedwoodLogs',
    'PersistDNSLogs',
)
