import csv
from lchttp.uri import decode
import logging
from logtailer.settings import (
    LC_CAT_PREFIX,  # noqa
    NON_SCORERS,  # noqa
)
from logtailer.text import flexi_csv_reader
from logtailer.typehints import AccessLogData, AccessLogWsData

logger = logging.getLogger(__name__)


# ---------------------------------------------------------------
def acl_dict(line: str) -> dict[str, int]:
    """
    Convert tally / score string to dictionary, where string is

    'lc_auto 23543, lc_racing 423, racing 233, sports 158'

    '< guitar> 815, <electric guitar> 244, <name search> 122, <logistics> 60'

    >>> acl_dict('lc_auto 23543, lc_racing 423, racing 233, sports 158')
    {'< guitar>': 815, '<electric guitar>': 244, '<name search>': 122, '<logistics>': 60}

    """
    if not line:
        return {}

    acl_data = {}

    for acl in line.split(','):
        acl = acl.strip()
        try:
            r_space = acl.rfind(' ')
            name = acl[:r_space]
            acl_data[name] = int(acl[r_space:])
        except (TypeError, ValueError):
            logger.error(f'{line} is malformed. Unable to parse text & score.')

    return acl_data


# ----------------------------------------------------------------------
def log_line_dicts(logs: list, fields) -> list[dict]:
    """Return list of dictionaries from log lines"""

    if isinstance(logs, str):
        logs = logs.splitlines()

    return flexi_csv_reader(csv.DictReader(logs, fields))


# ----------------------------------------------------------------------
def log_line_dict(log: str, fields) -> dict:
    """Return dictionary from log line"""

    return flexi_csv_reader(csv.reader([log], fields))[0]


# ----------------------------------------------------------------------
def top_score(conditions: list[str], scores: dict[str, int]) -> str:
    """
    Extract Top-scoring category & score, by looking at conditions
    and getting the highest interesting category there. If no interesting
    category found, then loop through scores and get first category
    of interest.
    """
    for acl in conditions:
        if acl in NON_SCORERS:
            continue

        try:
            return f'{acl} {scores[acl]}'
        except KeyError:
            break

    for cat, score in scores.items():
        if cat not in NON_SCORERS:
            return f'{cat} {score}'

    return ''


# ----------------------------------------------------------------------
def parse_access_log(ll: AccessLogData) -> AccessLogWsData:
    """
    Parse an access.log line before sending to WS server

    :param ll: dict from CSV-parsed Redwood access.log line
    :return: Dict of data ready to publish on WS server
    """
    if conditions := ll['conditions'] or '':
        score = top_score([c.strip() for c in conditions.split()], acl_dict(ll['scores']))
    else:
        score = ''
    ua = decode((ll['user_agent'] or '').strip()) or 'No UA Provided'
    page_title = ll['page_title'] or 'No Page Title'

    # If all categories aren't custom or line not blocked, hide rule Tally
    if not conditions.startswith(LC_CAT_PREFIX) or ll['action'] != 'block':
        tally = ''
    else:
        tally = ll['tally']

    base_log = '{user} {action} {url} {method} {content_type} (ACLs: {conditions})'.format(**ll)

    try:
        rule_description = ll['rule_description']
        record_prefix, pk = rule_description.split('/')
    except (AttributeError, KeyError, IndexError, ValueError):
        record_prefix = pk = ''

    return AccessLogWsData(
        userip=ll['user'],
        action=ll['action'],
        line=f'{ll["ldate"]} {base_log} "{ua}" "{page_title}" [{tally}] ({score})',
        pk=pk,
        record=record_prefix,
    )


__all__ = (
    'acl_dict',
    'log_line_dicts',
    'log_line_dict',
    'parse_access_log',
    'top_score',
)
