import logging
from pathlib import Path
from logtailer.settings import IS_DRAWBRIDGE_OS

if IS_DRAWBRIDGE_OS:
    # union-fs is a user-space file system, so kernel notify events
    # don't fire as often native file systems
    from watchdog.observers.polling import PollingObserver as TailObserver
else:
    from watchdog.observers import Observer as TailObserver  # type: ignore[assignment]

logger = logging.getLogger(__name__)


class LogTailerObserver:
    """
    Subclasses must define the "watch_file" attribute
    """

    def __init__(self, handler):
        self.observer = TailObserver()
        self.handler = handler

    def start(self):
        watch_file = self.handler.get_watch_file()
        watch_file_path = Path(watch_file)

        try:
            self.observer.schedule(self.handler, watch_file, recursive=watch_file_path.is_dir())
            self.observer.start()
            logger.info(f'Starting observer for {watch_file}')
        except FileNotFoundError:
            logger.info(f'Skipping observer for "{watch_file}". File not found.')
        except Exception as e:
            logger.error(f'Unable to start observer for "{watch_file}": {e}')

    def stop(self):
        self.observer.stop()

        try:
            self.observer.join(0.25)
        except RuntimeError:
            pass
        except Exception as e:
            logger.exception(f'Observer failed to stop: {self.__class__.__name__}: {e}')

        logger.info(f'{self.__class__.__name__} Observer stopped')


__all__ = [
    'LogTailerObserver',
]
