import logging
from typing import TYPE_CHECKING

from ..parsers import parse_access_log, skip_malformed
from ..persistence import PersistRedwoodLogs
from ..settings import (  # type: ignore[attr-defined]
    REDWOOD_LOG_DIR,
    REDWOOD_LOG_FIELDS,  # noqa
    TLS_LOG_FIELDS,  # noqa
)
from ..typehints import AccessLogData, AccessLogWsData, TlsLogData, TlsLogWsData

if TYPE_CHECKING:
    from ..handlers.base import LogTailerHandler

    MixinBase = LogTailerHandler
else:
    MixinBase = object

logger = logging.getLogger(__name__)


# ----------------------------------------------------------------------
class RedwoodAccessMixin(MixinBase):
    """
    Tail Redwood Access log
    """

    name = 'redwood'
    ws_channel = 'filter_access'
    offset_file = '/var/tmp/redwood.offset'
    watch_file = f'{REDWOOD_LOG_DIR}/access.log'
    persist_class = PersistRedwoodLogs
    csv_fields = REDWOOD_LOG_FIELDS

    def parsed_persistable_data(self, ll: AccessLogData, line: str) -> AccessLogData:  # type: ignore[override]
        """
        Return log line data if it's well-formed and passes "scrub" test.
        """
        if not skip_malformed(ll):
            return ll

    def parsed_ws_message(self, ll: AccessLogData, line: str) -> AccessLogWsData | None:  # type: ignore[override]
        """
        Scrub out all lines that shouldn't be sent to websocket server.
        Don't call scrub_logline here, as we need to send ssl-bump
        lines to the websocket server.
        """
        if not skip_malformed(ll):
            try:
                return parse_access_log(ll)
            except Exception as e:
                logger.error(f'Unable to parse Log Data {ll}. Error: {e}')


# ----------------------------------------------------------------------
class RedwoodErrorsMixin(MixinBase):
    """
    Tail Redwood Errors log
    """

    name = 'errors'
    ws_channel = 'filter_errors'
    offset_file = '/var/tmp/redwood_errors.offset'
    watch_file = f'{REDWOOD_LOG_DIR}/errors.log'
    persist_class = None
    csv_fields = ()

    def as_data(self, line: str) -> dict:
        return {'line': line}


# ----------------------------------------------------------------------
class RedwoodTLSMixin(MixinBase):
    """
    Tail Redwood TLS Log
    """

    name = 'tls'
    ws_channel = 'filter_tls'
    offset_file = '/var/tmp/redwood_tls.offset'
    watch_file = f'{REDWOOD_LOG_DIR}/tls.log'
    persist_class = None
    csv_fields = TLS_LOG_FIELDS

    def parsed_ws_message(self, ll: TlsLogData, line: str) -> TlsLogWsData:  # type: ignore[override]
        return TlsLogWsData(
            server=ll['serverAddr'],
            userip=ll['user'],
            line=line,
        )


__all__ = (
    'RedwoodAccessMixin',
    'RedwoodErrorsMixin',
    'RedwoodTLSMixin',
)
