from cent import Client, ClientNotEmpty, RequestException
import csv
import logging
import os
import signal
import time
from typing import Sequence
from watchdog.events import FileSystemEventHandler, FileSystemEvent
from ..persistence import PersistLogs
from ..pygtail import Pygtail
from ..settings import BUFFER_LINES, CLOUD_SERVER, FILTER_SERVER
from ..text import flexi_csv_generator

# use a larger threshold if cloud server, as the activity will be much higher
BUFFER_SIZE = BUFFER_LINES if CLOUD_SERVER else BUFFER_LINES / 2

logger = logging.getLogger(__name__)


# ----------------------------------------------------------------------
class LogTailerHandler(FileSystemEventHandler):
    """
    Base Log Tailer handler.

    When tailing a log file to save lines to database, we need
    to manage the last position so we can start there at the
    next launch time.
    """

    offset_file: str
    csv_fields: Sequence[str]
    persist_class: type[PersistLogs] | None
    watch_file: str
    ws_channel: str

    def __init__(self, ws_client: Client) -> None:
        self._last_position: int = 0
        self.ws_client: Client = ws_client
        self.persist_client = self.persist_class() if self.persist_class else None

        # on startup, assume no ws presence until _last_presence_check expires
        self._websocket_has_presence: bool = False
        self._last_presence_check: float = time.time()

        self.post_to_console_api = bool(self.persist_class) and FILTER_SERVER

        self._tailer: Pygtail | None = None

        # Buffer lines before POSTing to API handler
        self._api_buffer: list[dict] = []

    def on_closed(self, event: FileSystemEvent) -> None:
        self.persist_data(flush=True, update_offset=True)
        self.terminate_handler(event)

    def on_deleted(self, event: FileSystemEvent) -> None:
        self.persist_data(flush=True, update_offset=False)
        self.terminate_handler(event)

    def on_moved(self, event: FileSystemEvent) -> None:
        self.persist_data(flush=True, update_offset=False)
        self.terminate_handler(event)

    def on_modified(self, event: FileSystemEvent) -> None:
        """
        Read all new lines on receiving the "modified" event.
        """

        # sleep a bit to allow more lines to accumulate
        time.sleep(0.3)

        for line in self.tailer:
            lld = self.as_data(line)
            if not lld:
                continue
            self.publish_to_websocket(lld, line)
            self.add_to_buffer(lld, line)

    def terminate_handler(self, event: FileSystemEvent) -> None:
        """
        Catch deletes. The task manager should restart to watch the new inode.
        """
        pid = os.getpid()
        logger.info(f'Terminating PID: {pid} - received event: {event}')
        os.kill(pid, signal.SIGTERM)

    def parsed_ws_message(self, ll: dict, line: str) -> dict:
        """
        Override on subclasses to customize parsing and the lines included.
        """
        return ll

    def websocket_has_presence(self) -> bool:
        """
        If websocket server has no user present, then there's
        no reason to publish messages to the server.

        Check for presence only every 5 seconds as modified
        events will occur much oftener than that.
        """
        # if no channel is specified, then we're not publishing messages
        if not self.ws_channel:
            return False

        if (now := time.time()) - self._last_presence_check > 5:
            self._last_presence_check = now
            try:
                self._websocket_has_presence = self.ws_client.presence(self.ws_channel)
            except ClientNotEmpty:
                self._websocket_has_presence = True
            except RequestException:
                self._websocket_has_presence = False

        return self._websocket_has_presence

    def publish_to_websocket(self, ll: dict, line: str) -> None:
        """
        Publish log lines to websocket server.
        """
        if not self.websocket_has_presence():
            return

        if ll := self.parsed_ws_message(ll, line):
            try:
                self.ws_client.publish(self.ws_channel, ll)
            except Exception as e:
                logger.exception(f'Unable to publish {line!r} to websocket: {e}')

    def parsed_persistable_data(self, ll: dict, line: str) -> dict:
        """
        Override on subclasses to customize parsing and the lines included.
        """
        return ll

    def add_to_buffer(self, ll: dict, line: str) -> None:
        """
        Accumulate log lines and then post console
        task scheduler to save them to the database.
        """
        if not self.post_to_console_api:
            return

        if not (parsed := self.parsed_persistable_data(ll, line)):
            return

        self._api_buffer.append(parsed)
        self.persist_data()

    def persist_data(self, flush: bool = False, update_offset: bool = True) -> None:
        """
        Persist all the lines in the buffer.

        :param flush: Persist lines even if the buffer isn't full.
        :param update_offset: Update the offset file position. Should
            not be done if the file was moved or deleted.
        """
        if flush or len(self._api_buffer) > BUFFER_SIZE:
            if self.persist_client:
                buffered_lines = self._api_buffer.copy()
                self._api_buffer = []
                self.persist_client.save(buffered_lines)

            if update_offset:
                self.tailer.update_offset_file()

    def get_watch_file(self) -> str:
        """
        Override on subclasses in case any special logic is required.
        Basically for redwood's errors.log file which doesn't close
        when log-rotated, so we need to find the opened file.
        """
        return self.watch_file

    def as_data(self, line: str) -> dict:
        """
        Parse a line and return it as dictionary.
        Override on subclass mixins to customize.
        """
        try:
            return flexi_csv_generator(
                csv.DictReader((line,), fieldnames=self.csv_fields),
            ).__next__()
        except (RuntimeError, StopIteration) as e:
            logger.info(f'Malformed logline: {line}')
            logger.exception(e)
            return {}

    @property
    def tailer(self) -> Pygtail:
        """
        Open file object that reads new log lines.
        """
        if not self._tailer:
            self._tailer = Pygtail(
                self.get_watch_file(),
                every_n=BUFFER_LINES,
                offset_file=self.offset_file or None,
                read_from_end=not bool(self.offset_file),
                full_lines=True,
            )
        return self._tailer


__all__ = [
    'LogTailerHandler',
]
