from cent import Client
import logging
import os
import signal
import time

from .settings import CENTRIFUGE_API_KEY, TMP_DIR, WEBSOCKET_PORT

from .observers import (
    RedwoodAccessObserver,
    RedwoodErrorsObserver,
    RedwoodTLSObserver,
    PDNSRecursorObserver,
)

logger = logging.getLogger('logtailer')


def main():
    os.makedirs(f'{TMP_DIR}/reporter/logline', exist_ok=True)
    all_observers = []

    class ProcessExit(Exception):
        """
        Custom exception which is used to trigger the clean exit
        of all running threads and the main program.
        """

        pass

    def service_shutdown(signum, frame):
        logger.info('Caught signal %d' % signum)
        raise ProcessExit

    # Register the signal handlers
    signal.signal(signal.SIGTERM, service_shutdown)
    signal.signal(signal.SIGINT, service_shutdown)
    signal.signal(signal.SIGHUP, service_shutdown)

    ws_client = Client(
        address=f'http://localhost:{WEBSOCKET_PORT}',
        api_key=CENTRIFUGE_API_KEY,
    )

    try:
        for observer in (
            RedwoodAccessObserver,
            RedwoodErrorsObserver,
            RedwoodTLSObserver,
            PDNSRecursorObserver,
        ):
            all_observers.append(observer(ws_client))

        for observer in all_observers:
            logger.info(f'starting observer {observer}')
            observer.start()

        # Keep the main thread running, otherwise signals are ignored.
        try:
            while True:
                time.sleep(1.0)
        except (ProcessExit, KeyboardInterrupt):
            pass

    except Exception as e:
        logger.exception(e)

    logger.info('Shutdown Log Tailers')
    for observer in all_observers:
        try:
            observer.stop()
        except (ProcessExit, KeyboardInterrupt):
            pass


if __name__ == "__main__":
    main()
