#!/usr/bin/python3

import io
import re
import os
from os.path import dirname, join
from setuptools import setup


def read(*names, **kwargs):
    path = join(dirname(__file__), *names)
    return io.open(path, encoding=kwargs.get('encoding', 'utf8')).read()


def get_packages(package):
    """
    Return root package and all sub-packages.
    """
    return [
        dirpath
        for dirpath, dirnames, filenames in os.walk(package)
        if os.path.exists(os.path.join(dirpath, '__init__.py'))
    ]


def get_package_data(package):
    """
    Return all files under the root package, that are not in a
    package themselves.
    """
    walk = [
        (dirpath.replace(package + os.sep, '', 1), filenames)
        for dirpath, dirnames, filenames in os.walk(package)
        if not os.path.exists(os.path.join(dirpath, '__init__.py'))
    ]

    filepaths = []
    for base, filenames in walk:
        filepaths.extend([os.path.join(base, filename) for filename in filenames])
    return {package: filepaths}


def get_version():
    with open('logcabinctl/__init__.py', 'r') as f:
        line = f.readline()
        match = re.match(r'__version__ = \'([\d\.]+)\'', line)

        if not match:
            raise ImportError("Can't read the version of Log Cabin Ctl")

        version = match.group(1)
        return version


setup(
    name='logcabinctl',
    version=get_version(),
    license='SAP',
    description='Log Cabin Console CLI Control Center',
    long_description_content_type='text/markdown',
    author='Dave Burkholder',
    author_email='dave@compassfoundation.io',
    url='https://code.compassfoundation.io/dave/logcabinctl',
    packages=get_packages('logcabinctl'),
    package_data=get_package_data('logcabinctl'),
    # py_modules=[splitext(basename(path))[0] for path in glob('src/*.py')],
    zip_safe=True,
    options={
        'bdist_rpm': {
            'post_install': 'post_install.sh',
            'pre_install': 'pre_install.sh',
        },
    },
    classifiers=[
        'Intended Audience :: Developers',
        'License :: SAP',
        'Operating System :: Unix',
        'Operating System :: POSIX',
        'Programming Language :: Python :: 3.11',
        'Programming Language :: Python :: 3.12',
        'Programming Language :: Python :: Implementation :: CPython',
        'Programming Language :: Python :: Implementation :: PyPy',
        'Topic :: Utilities',
    ],
    keywords=[
        'Draw Bridge',
        'Log Cabin',
    ],
    install_requires=[
        'aspen_crypto',
        # 'capsule',
        'console_base',
        'invoke',
        'lcconfig',
        'lcrequests',
        'psutil',
    ],
    entry_points={
        'console_scripts': [
            'logcabin = logcabinctl.main:program.run',
            'drawbridge = logcabinctl.main:program.run',
            'portal = logcabinctl.main:program.run',
        ],
    },
)
