from aspen_crypto.keys import (
    Ed25519PrivateKey,
    load_local_umbrella_api_key,
    persist_umbrella_api_key,
)
from aspen_crypto.settings import ENCRYPTION_SECTION, LOCAL_UMBRELLA_KEY_CID
from lcconfig import ConsoleSettingsConfig
from lcrequests.exceptions import LCHTTPError
from lcrequests.typehints import Subject
import pybase64
from pylogcabin.umbrella import CompassUmbrellaAPI
from pylogcabin.tokens import DrawBridgePortalToken, V4PasetoToken
from typing import Literal, overload
from logcabinctl.typehints import CanonicalID, ConfigData, Impersonation


def get_drawbridge_config_data(cid: CanonicalID, token: V4PasetoToken | None = None) -> ConfigData:
    """
    Retrieve DrawBridge configuration data from Odoo.
    """
    return get_config_data('config/register', cid, token)


def get_impersonation_config_data(
    cid: CanonicalID,
    token: V4PasetoToken | None = None,
) -> Impersonation:
    """
    Retrieve system data from Odoo that to enable provisioning
    this system as an impersonator of the failed Appliance.
    """
    return get_config_data('impersonate', cid, token)


# -------------------------------------------------------------------------
@overload
def get_config_data(
    path: Literal['impersonate'],
    cid: CanonicalID,
    token: V4PasetoToken | None = None,
) -> Impersonation: ...


@overload
def get_config_data(
    path: Literal['config/register'],
    cid: CanonicalID,
    token: V4PasetoToken | None = None,
) -> ConfigData: ...


def get_config_data(
    path: str,
    cid: CanonicalID,
    token: V4PasetoToken | None = None,
) -> ConfigData | Impersonation:
    """
    Retrieve DrawBridge Appliance configuration data from Odoo.
    """
    try:
        api = CompassUmbrellaAPI(load_token(token))
        api.load(Subject.DrawBridge, path)
        resp = api.post(cid)
        return resp.json()

    except LCHTTPError as e:
        msg = e.response.json()
        raise SystemExit(f'Unable to retrieve data from {path!r} - Msg: \n\n{msg}\n') from None
    except Exception as e:
        raise SystemExit(f'Unable to contact Odoo - Error: \n\n{e}\n') from None


# -------------------------------------------------------------------------
def load_token(token: V4PasetoToken | None, user_cid: CanonicalID | None = None) -> V4PasetoToken:
    """
    If token is not specified, load the local umbrella token or pinned token.
    """
    if token:
        return token

    api_key = load_local_umbrella_api_key()

    return DrawBridgePortalToken(api_key.key, user_cid=user_cid, verification_cid=api_key.cid)


def register_umbrella_api_key(cid: CanonicalID) -> dict:
    """
    If the API Key hasn't been generated, create an
    Umbrella API Private Key and register it with
    Odoo Umbrella server.
    """
    if umbrella_key_registered():
        return {}

    private_key = Ed25519PrivateKey.generate()
    public_key_pem = pybase64.urlsafe_b64encode(private_key.public_key.as_pem)
    token = DrawBridgePortalToken(private_key)
    path = 'aspen/register'

    try:
        api = CompassUmbrellaAPI(token)
        api.load(Subject.DrawBridge, path)
        resp = api.post(cid, post_data={'pem': public_key_pem})
        data = resp.json()
        persist_umbrella_api_key(key_cid=data['cid'], private_key=private_key)
        return data

    except LCHTTPError as e:
        msg = e.response.json()
        raise SystemExit(f'Unable to retrieve data from {path!r} - Msg: \n\n{msg}\n') from None
    except Exception as e:
        raise SystemExit(f'Unable to contact Odoo - Error: \n\n{e}\n') from None


def retire_umbrella_api_key() -> dict:
    """
    Retire Umbrella Public Key on Odoo Umbrella server.
    """
    path = 'aspen/retire'
    data = ConsoleSettingsConfig().as_typed_dict(ENCRYPTION_SECTION)
    if not (cid := data.get(LOCAL_UMBRELLA_KEY_CID)):
        print('No Umbrella Verification Key found to retire')
        return {}

    try:
        api = CompassUmbrellaAPI(load_token(token=None))
        api.load(Subject.DrawBridge, path)
        resp = api.post(cid)
        load_local_umbrella_api_key.cache_clear()
        return resp.json()
    except Exception as e:
        print(f'Unable to retire Umbrella Key - {e}')


def umbrella_key_registered() -> bool:
    """
    If the umbrella key has already been registered,
    the CID will be present in the config file.
    """
    data = ConsoleSettingsConfig().as_typed_dict(ENCRYPTION_SECTION)
    return bool(data.get(LOCAL_UMBRELLA_KEY_CID))


__all__ = (
    'get_drawbridge_config_data',
    'get_impersonation_config_data',
    'register_umbrella_api_key',
    'retire_umbrella_api_key',
    'umbrella_key_registered',
)
