import os
import socket
import sys

from .filesize import readable_size
from pathlib import PosixPath
import psutil

from ..settings import CONSOLE_HOME, PORTAL_USER, PORTAL_CODE_NAME, PORTAL_LOG_DIR, DB_HOST


# --------------------------------------------------------------------------------
def check_disk() -> None:
    """
    Print out disk utilization percent
    """
    print('Checking Available Disk Space')
    space = psutil.disk_usage('/').percent

    prefix = ''
    if space > 90 < 95:
        prefix = 'WARNING!'
    if space >= 95:
        prefix = 'DISASTER!'

    print(f'{prefix} Hard drive space is {space}% utilized')


# --------------------------------------------------------------------------------
def check_file_ownership() -> None:
    """
    ClearOS 5 systems have an annoying bug where system uids vanish unless
    the user is created in ldap. Check to see that file owners have correct uid
    """

    def _verify_owner(path: PosixPath, owner: str = PORTAL_USER) -> None:
        """Verify that a file / directory is owned by the specified owner"""
        if not path.exists():
            print(f'DISASTER! {path} does not exist')
            return
        try:
            if not path.owner() == owner:
                print(f'WARNING! {path} is not owned by {owner}')
        except KeyError:
            print(f'DISASTER! {path} uid is not found in the system database')

    print('Checking File Ownership')
    log_path = PosixPath(PORTAL_LOG_DIR)

    _verify_owner(log_path)

    for log_file in log_path.glob('*.log'):
        _verify_owner(log_file)


# --------------------------------------------------------------------------------
def check_postgres() -> None:
    _check_database(5432, 'PostgresQL')


# --------------------------------------------------------------------------------
def _check_database(port: int, name: str) -> None:
    """
    Ensure that database connections can be made
    """
    print(f'Checking {name} Status')
    try:
        from psycopg import connect as db_connect

        conn = db_connect(
            f"dbname='{PORTAL_CODE_NAME}' user='{PORTAL_CODE_NAME}' host={DB_HOST} port={port}"
        )
    except Exception:  # noqa
        print(f'DISASTER! Unable to connect to database via {name}. Server may not be running.')
        return

    if conn.closed == 0:
        print('Successfully connected to database')
        conn.close()
    else:
        print('DISASTER! Unable to connect to database')


# --------------------------------------------------------------------------------
def check_process(pname: str) -> None:
    """
    Check to see if process is running,
    and if so, calculate RAM / CPU stats.

    :param str pname: Process Name
    :return:
    """
    processes = [p for p in psutil.process_iter() if p.name() == pname]
    if not processes:
        print(f'DISASTER! {pname} is not running')
        return

    try:
        cpu = 0
        mem = 0
        for p in processes:
            cpu += p.cpu_percent()
            mem += p.memory_full_info().uss
    except psutil.AccessDenied:
        print(f'WARNING! {pname} is running, but unable to retrieve stats as current user')
        return

    print(
        'SUCCESS! {} is running and consuming {} of RAM and {:.2f}% of CPU'.format(
            pname,
            readable_size(mem),
            cpu,
        )
    )


# --------------------------------------------------------------------------------
def check_http_reachable() -> None:
    """
    Make HTML hit to console and confirm if services are reachable.
    """
    os.chdir(CONSOLE_HOME)
    sys.path.append(CONSOLE_HOME)
    from healthcheck import HealthCheck  # noqa

    healthcheck = HealthCheck()
    resp = healthcheck.check(notify=False)

    if healthcheck.all_ok():
        print('SUCCESS! Console reachable by browser and all systems OK')
        return

    if resp:
        print(f'WARNING! Console Health Check returns {resp}')
    else:
        print('DISASTER! Console unreachable via http request')


def check_port_reachable(port: int, name: str) -> None:
    """
    Check that specified port is responding.
    """
    try:
        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        s.connect(('localhost', port))
    except Exception:  # noqa
        print(f'DISASTER! {name} is not running')
    finally:
        s.close()  # noqa


__all__ = (
    'check_file_ownership',
    'check_disk',
    'check_postgres',
    'check_process',
    'check_http_reachable',
    'check_port_reachable',
)
