from aspen_crypto.keys import load_private_key
from aspen_crypto.encode import mask
from capsule.wizards import persist_object_storage_config
from console_keyring.console_keyring import ConsoleKeyring
from lcconfig.configparser import (
    CareCenterSettingsConfig,
    ClavisAuthSettingsConfig,
    ConsoleSettingsConfig,
    ConsoleImpersonatorConfig,
    DynamicDNSConfig,
    ObjectStorageConfig,
    SyncPublisherConfig,
    SymmetricKeysConfig,
)
from lcconfig.yamlconfig import (
    AuthLdapYmlConfig,
    AuthActiveDirectoryYmlConfig,
    ConcordiaYmlConfig,
)
from lchttp.json import json_dumps, json_loads
from lcutils import run_cmd
import logging
from pathlib import Path
from pybase64 import urlsafe_b64encode
import secrets
from ..settings import (
    CONF_DIR,
    EMAIL_SERVER_HOST,
    EMAIL_SERVER_PORT,
    SYSTEM_CFG_BLOB_FILE,
    TLS_DIR,
    VERIFICATION_KEY_FILE,
)
from ..typehints import ConfigData, SyncPublisherData

logger = logging.getLogger(__name__)


def create_symmetric_key_config() -> dict:
    """
    Create the Symmetric Keys config file.
    """
    symmetric_keys = SymmetricKeysConfig()
    cfg = symmetric_keys.as_typed_dict()
    chacha_key = cfg.get('chacha')

    if not chacha_key:
        chacha_key = mask(secrets.token_urlsafe(24))
        cfg['chacha'] = chacha_key
        symmetric_keys.save_section(section=SymmetricKeysConfig.main_section, data=cfg)

    kr = ConsoleKeyring()

    # On new systems, the password file won't exist.
    try:
        fernet_key = kr.get_password('console', 'fernet')
    except FileNotFoundError:
        fernet_key = None

    if not fernet_key:
        fernet_key = urlsafe_b64encode(secrets.token_urlsafe(24).encode()).decode()
        kr.set_password('console', 'fernet', fernet_key)

    return {
        'fernet_key': fernet_key,
        'chacha': chacha_key,
        'private_key_password_prefix': load_private_key('chacha'),
    }


def persist_ddns_url(local_url: str, impersonated_url: str = '') -> None:
    """
    Update the DDNS URL(s).

    :param local_url: The DDNS URL of the local console.
    :param impersonated_url: The DDNS URL of the system that is being impersonated.
    """
    new_data = {}
    cfg = DynamicDNSConfig()
    dns_cfg_data = cfg.as_typed_dict()

    if local_url and impersonated_url:
        if dns_cfg_data.get('impersonator_url') != local_url:
            new_data['impersonator_url'] = local_url
        if dns_cfg_data.get('url') != impersonated_url:
            dns_cfg_data['url'] = impersonated_url

    elif dns_cfg_data.get('url') != local_url:
        new_data['url'] = local_url

    if not new_data:
        return

    dns_cfg_data |= new_data
    cfg.save_section(data=dns_cfg_data)

    logger.info(f'Saved DDNS URL as {impersonated_url or local_url} in Dynamic DNS Config')


def persist_sync_publisher_config(publisher_data: SyncPublisherData) -> SyncPublisherData:
    """Update the Sync Publisher config file"""
    sync_server_cfg = SyncPublisherConfig()
    if not publisher_data or publisher_data == sync_server_cfg.as_typed_dict():
        return publisher_data

    sync_server_cfg.save_section(data=publisher_data)
    logger.info('Updated Sync Publisher configuration file')

    return publisher_data


def persist_local_appliance_config(cfg_data: ConfigData) -> None:
    """
    Save local appliance data to the default settings.conf location.
    """
    cfg = ConsoleSettingsConfig()
    cfg.save_section('BoxData', load_config_data(cfg_data))
    cfg.save_email(load_email_data(cfg_data))


def persist_impersonator_config(cfg_data: ConfigData) -> None:
    """
    Save impersonator.conf file for local settings when an Appliance is
    impersonating another appliance as a warm spare.

    The impersonated system's data will get saved to the default location,
    and the impersonator data to another location, so that we can tell the
    system is impersonating another system rather than being a self-standing
    unit. But the impersonator should still be reachable via its own DNS names
    and the VPN settings and Redwood CAs should be issued to the impersonator.
    """
    data = load_config_data(cfg_data)

    data |= {
        'api_username': f'{cfg_data["name"]}_api',
        'api_user_cid': f'{cfg_data["cid"]}',
        'accountable': False,
        'filter_server': False,
    }

    cfg = ConsoleImpersonatorConfig()
    cfg.save_section('BoxData', data)
    cfg.save_email(load_email_data(cfg_data))


def load_config_data(cfg_data: ConfigData) -> dict:
    """
    Extract needed fields from Odoo registration data dict.
    """
    data = {}
    for field in (
        'adminurl',
        'cid',
        'external_port',
        'device_dns_name',
        'name',
        'rebranded',
        'theme',
    ):
        data[field] = cfg_data.get(field)

    if not data:
        return data

    data |= {
        'api_username': f'{cfg_data["name"]}_api',
        'api_user_cid': f'{cfg_data["cid"]}',
        'accountable': False,
        'filter_server': False,
    }
    return data


def load_email_data(cfg_data: ConfigData) -> dict:
    """
    Extract fields needed from Odoo data dict for email settings.
    """
    email_data = {
        'port': EMAIL_SERVER_PORT,
        'host': EMAIL_SERVER_HOST,
    }
    for odoo_field, email_field in (
        ('email', 'username'),
        ('email', 'from_address'),
        ('email_password', 'password'),
    ):
        email_data[email_field] = cfg_data[odoo_field]  # type: ignore

    return email_data


def persist_garage_storage_config(odoo_data: ConfigData, service: str) -> ConfigData:
    """Update the S3 config file and secret"""

    if not odoo_data:
        return odoo_data

    creds = {}
    for odoo_key, cfg_key in (
        ('garage_key_secret', 'secret_key'),
        ('garage_key_id', 'access_key'),
    ):
        if new_value := odoo_data.get(odoo_key):  # noqa
            creds[cfg_key] = new_value

    persist_object_storage_config(
        endpoint=odoo_data['garage_endpoint'],
        region=odoo_data['garage_region'],
        creds=creds,
        service=service,
    )

    return odoo_data


def persist_config_data_blob(odoo_data: ConfigData):
    """
    Save raw config data to disk, so we can load it in
    a Django management command and save values to the
    portal database.
    """
    with open(f'{CONF_DIR}/{SYSTEM_CFG_BLOB_FILE}', 'wb') as df:
        df.write(json_dumps(odoo_data))


def load_config_data_blob() -> ConfigData | dict:
    """
    Try loading config data from local file.

    This function will be used in the portal side
    to load config data to save to the database.

    If file is not found, then it'll attempt a
    registration API call.
    """
    try:
        with open(f'{CONF_DIR}/{SYSTEM_CFG_BLOB_FILE}', 'rb') as df:
            return json_loads(df.read())
    except (FileNotFoundError, ValueError):
        pass

    return {}


def delete_config_data_blob() -> None:
    """
    The Config data blob should be deleted
    after it's loaded into the console!

    Helper method to be called from console
    and `delete_config_files` function.
    """
    blob_data_file = f'{CONF_DIR}/{SYSTEM_CFG_BLOB_FILE}'
    try:
        Path(blob_data_file).unlink(missing_ok=True)
    except Exception as e:
        print(f'Unable to remove {blob_data_file}. Error: {e}')


def delete_config_files() -> None:
    """
    Delete all config fields that are specific to an Appliance.
    """
    delete_config_data_blob()
    kr = ConsoleKeyring()
    for cfg_file in (
        kr.file_path,
        CareCenterSettingsConfig.persist_file,
        ClavisAuthSettingsConfig.persist_file,
        ConsoleSettingsConfig.persist_file,
        ConsoleImpersonatorConfig.persist_file,
        DynamicDNSConfig.persist_file,
        ObjectStorageConfig.persist_file,
        SymmetricKeysConfig.persist_file,
        AuthActiveDirectoryYmlConfig.conf_file,
        AuthLdapYmlConfig.conf_file,
        ConcordiaYmlConfig.conf_file,
        VERIFICATION_KEY_FILE,
        f'{TLS_DIR}/concord_mq_client.csr',
        f'{TLS_DIR}/concord_mq_client.key',
        f'{TLS_DIR}/concord_mq_client.pem',
        f'{TLS_DIR}/console.key',
        f'{TLS_DIR}/console.pem',
        f'{TLS_DIR}/redwood_ca.pem',
        f'{TLS_DIR}/redwood_ca.key',
        f'{CONF_DIR}/passwd.txt',
    ):
        try:
            Path(cfg_file).unlink(missing_ok=True)
        except PermissionError:
            run_cmd(f'sudo rm -f {cfg_file}')


__all__ = (
    'create_symmetric_key_config',
    'persist_ddns_url',
    'persist_local_appliance_config',
    'persist_impersonator_config',
    'persist_sync_publisher_config',
    'persist_garage_storage_config',
    'persist_config_data_blob',
    'load_config_data_blob',
    'delete_config_data_blob',
    'delete_config_files',
)
