from console_base.utils import value_to_hex
from cryptography.hazmat.backends import default_backend
from cryptography.x509.base import load_pem_x509_certificate
from invoke import Context, task
from lcconfig.configparser import ConsoleSettingsConfig
from lcrequests.exceptions import LCHTTPError
from pylogcabin.client import AspenAPI

from .sys import restart_webserver
from ..settings import (
    CONSOLE_HOME,
    IS_DRAWBRIDGE_OS,
    PYTHON,
    REDWOOD_CA_CERT_FILE,
)
from ..utils.network import load_token


@task(post=[restart_webserver], help={'force': 'Refresh TLS certificate whether stale or not'})
def refresh_console_cert(ctx: Context, force=False):
    """
    Refresh Console web server TLS certificate
    """
    print('Refresh Console TLS certificate')
    cmd = f'{PYTHON} {CONSOLE_HOME}/manage.py update_console_cert'

    if force:
        cmd = f'{cmd} --force'

    if IS_DRAWBRIDGE_OS:
        ctx.sudo(cmd, pty=True, user='portal')
    else:
        ctx.run(cmd, pty=True)


@task(help={'force': 'Refresh Redwood CA whether stale or not'})
def refresh_redwood_ca(ctx: Context, force=False):
    """
    Refresh Redwood CA
    """
    print('Refresh Redwood certificate authority')
    cmd = f'{PYTHON} {CONSOLE_HOME}/manage.py renew_redwood_ca'

    if force:
        cmd = f'{cmd} --force'

    if IS_DRAWBRIDGE_OS:
        ctx.sudo(cmd, pty=True, user='portal')
    else:
        ctx.run(cmd, pty=True)


@task
def retire_redwood_ca(ctx: Context):
    """
    Retire a Redwood CA; when:
        * A appliance is retired
        * Appliance hardware is replaced with new hardware
        * A warm spare appliance is reset, and traffic is handled by primary appliance
    """
    try:
        with open(REDWOOD_CA_CERT_FILE, 'rb') as cf:
            ca = load_pem_x509_certificate(
                cf.read(),
                backend=default_backend(),
            )
    except FileNotFoundError:
        print('No Redwood certificate authority found. Confirm CA is retired in Aspen.')
        return

    print('Retire Redwood certificate authority')
    cfg = ConsoleSettingsConfig().as_typed_dict('BoxData')
    system_cid = user_cid = cfg['cid']
    aspen_api = AspenAPI(signed_token_class=load_token(None, user_cid=user_cid))
    aspen_api.load('accounts', 'appliance')

    try:
        response = aspen_api.put(
            system_cid,
            subpath='retire_redwood_ca',
            post_data={'serial': value_to_hex(ca.serial_number)},
        )
        data = response.json()
        print(f'Redwood Redwood CA: {data.get("message")}')
    except LCHTTPError as e:
        msg = e.response.json().get('message')
        raise SystemExit(f'Unable to retire Redwood CA - Msg: \n\n{msg}\n') from None
    except Exception as e:
        raise SystemExit(f'Unable to retire Redwood CA: \n\n{e}\n') from None


__all__ = (
    'refresh_console_cert',
    'refresh_redwood_ca',
    'retire_redwood_ca',
)
