from invoke import Context, task

from ..utils.health import (
    check_file_ownership,
    check_process,
    check_postgres,
    check_disk,
    check_http_reachable,
    check_port_reachable,
)
from ..utils.osutils import get_database_service_name
from logcabinctl.settings import IS_DRAWBRIDGE_OS


@task
def health(ctx: Context):
    """
    Check overall system health of Log Cabin Console processes
    """
    for func in (
        check_file_ownership,
        check_disk,
        check_postgres,
    ):
        print()
        func()

    psql_process = 'postgres' if IS_DRAWBRIDGE_OS else 'postmaster'
    for process in ('nginx', psql_process, 'redis-server'):
        check_process(process)

    check_port_reachable(port=1517, name='wsgi server')
    check_port_reachable(port=10009, name='ivylink')

    check_http_reachable()


@task
def restart(ctx: Context):
    """Restart the Log Cabin Console and Scheduler"""
    ctx.run('service supervisord restart')


@task(post=[restart])
def nuclear_restart(ctx: Context):
    """
    Force restart all processes associated with the Log Cabin console

    Do not run unless 'health' indicates memory usage is high,
    or a simple 'restart' didn't restore console functionality.
    """
    if IS_DRAWBRIDGE_OS:
        processes = ['nginx', 'redis', 'portal-server', 'portal-cluster', 'portal-beat']
    else:
        processes = ['nginx', 'redis', 'supervisord']
    db_service_name = get_database_service_name()
    if db_service_name:
        processes.insert(0, db_service_name)

    for process in processes:
        ctx.run(f'service {process} restart')


@task
def restart_logtailer(ctx: Context):
    """Restart the Real Time Log Viewer stream"""
    ctx.run('supervisorctl restart logtailer')


@task
def restart_webserver(ctx: Context):
    """Restart the Console Web Server"""
    ctx.run('service nginx restart')


__all__ = (
    'health',
    'restart',
    'nuclear_restart',
    'restart_logtailer',
    'restart_webserver',
)
