from invoke import Context, task
from lcconfig import (
    ConsoleSettingsConfig,
    ConsoleImpersonatorConfig,
)
from lcutils import run_cmd
from capsule.utils.databases import create_console_database, drop_database
from capsule.settings import S3_REGION
from capsule.tasks.db import restore_all
from capsule.typehints import S3_REGIONS
from redis import Redis
from uuid import UUID
from ..settings import (
    CONSOLE_HOME,
    DATABASE_NAME,
    DRAWBRIDGE_PROCESSES,
    PYTHON,
)
from ..typehints import CanonicalID, ConfigData, Impersonation, KeyringService
from ..utils import (
    delete_config_files,
    get_drawbridge_config_data,
    get_impersonation_config_data,
    persist_ddns_url,
    persist_garage_storage_config,
    persist_local_appliance_config,
    persist_impersonator_config,
    persist_sync_publisher_config,
    persist_config_data_blob,
    create_symmetric_key_config,
    register_umbrella_api_key,
    retire_umbrella_api_key,
)


@task(
    help={
        'cid': (
            'Canonical ID of the Appliance to be registered. '
            '(Required when performing initial setup.)'
        ),
    },
)
def register(ctx: Context, cid: CanonicalID = ''):  # noqa
    """
    Perform initial registration and setup of an Appliance.
    """
    if cid and not isinstance(cid, UUID):
        try:
            UUID(cid)
        except (AttributeError, ValueError, TypeError):
            raise SystemExit(f'\n{cid!r} is not a valid Canonical ID. Please try again.\n')

    elif not cid:
        cfg = ConsoleSettingsConfig().as_typed_dict('BoxData')
        if not (cid := cfg.get('cid')):
            raise SystemExit('Please provide a Canonical ID by passing in "--cid <c.i.d.>')

    if ctx.config.get('skip-database-creation') != 'True':
        if create_console_database().created:
            cmd = f'{PYTHON} {CONSOLE_HOME}/manage.py migrate'
            print(f'Database was created. Running {cmd!r}')
            run_cmd(cmd)

    print('\n# ---------------------------------------------------------------')
    print('Retrieving Appliance data from Odoo\n')
    register_umbrella_api_key(cid)
    create_symmetric_key_config()
    cfg_data = get_drawbridge_config_data(cid)
    _save_local_appliance_config_data(cfg_data)

    if ctx.config.get('from-management-command') != 'True':
        _run_console_update_script(ctx)

    # Return the data, so that if this function is called from
    # within the Console, that the raw data can be accessed.
    return cfg_data


@task
def reset(ctx: Context):
    """
    Reset console by deleting database and config files.
    To restore functionality, run commands to re-register
    or impersonate another appliance.
    """
    Redis().flushdb()

    print('\nResetting the Console will delete the database and clear config files!\n')
    proceed = input('Are you sure you want to continue? [no/yes] ')

    if proceed.lower() == 'no':
        raise SystemExit('\nExiting Console Reset\n')

    if proceed.lower() != 'yes':
        raise SystemExit(f'\n{proceed} is an invalid value\n')

    from .tls import retire_redwood_ca

    try:
        retire_redwood_ca(ctx)
    except (BaseException, Exception) as e:
        print(e)

    print('Stopping filtering & portal services')
    run_cmd(f'systemctl stop {DRAWBRIDGE_PROCESSES}')
    # TODO figure out how to configure polkit to allow this command
    # run_cmd(f'systemctl disable {DRAWBRIDGE_PROCESSES}')

    retire_umbrella_api_key()

    print(f'Deleting database: {DATABASE_NAME}')
    drop_database(DATABASE_NAME)

    print('Deleting configuration files')
    delete_config_files()


@task(
    help={
        'cid': 'Canonical ID of the Appliance that will be restored from backups.',
        'region': f'Region for saving the archive file. Default: "{S3_REGION}"',
    },
)
def restore(ctx: Context, cid: CanonicalID, region: S3_REGIONS = S3_REGION):  # noqa
    """
    Restore all databases for this appliance from S3 backup.
    """
    ctx.config['skip-database-creation'] = 'True'
    register(ctx, cid)
    restore_all(ctx, region=region)


@task(
    help={
        'cid': (
            'Canonical ID of the failed Appliance that will be impersonated by this Appliance. '
            'The Odoo record must define the "Device DNS Target" of the desired spare Appliance.'
        ),
    },
    pre=(reset,),
)
def impersonate(ctx: Context, cid: CanonicalID):  # noqa
    """
    This appliance will impersonate another filter console on a temporary basis.

    This appliance cache and database will be reset, and a backup from the failed
    appliance restored on this system. Do not use this command unless you understand
    the implications!
    """
    if not isinstance(cid, UUID):
        try:
            UUID(cid)
        except (AttributeError, ValueError, TypeError):
            raise SystemExit(f'\n{cid!r} is not a valid Canonical ID. Please try again.\n')

    impersonator_data = ConsoleImpersonatorConfig().as_typed_dict('BoxData')
    console_data = ConsoleSettingsConfig().as_typed_dict('BoxData')

    if 'name' in impersonator_data and console_data.get('cid') == cid:
        raise SystemExit(
            f'{impersonator_data["name"]} is already impersonating {console_data["name"]}'
        )

    print('\n# ---------------------------------------------------------------')
    print('Retrieving Appliance data from Odoo\n')
    register_umbrella_api_key(cid)
    create_symmetric_key_config()
    cfg_data = get_impersonation_config_data(cid)
    _save_impersonation_config_data(cfg_data)

    # TODO - figure out how to configure this with polkit
    # run_cmd(f'systemctl enable {DRAWBRIDGE_PROCESSES}')


def _run_console_update_script(ctx: Context):
    cmd = f'{PYTHON} {CONSOLE_HOME}/logcabin_upgrades.py'
    print(f'Run console update script: {cmd!r}')
    # Use ctx.run so it exits if the command isn't successful
    ctx.run(cmd)


def _save_local_appliance_config_data(cfg_data: ConfigData):
    """
    Helper function to save configuration data for local appliance.
    """
    name = cfg_data['name']

    print(f'Saving Console settings for {name}')
    persist_local_appliance_config(cfg_data)

    print(f'Saving Dynamic DNS settings for {name}')
    persist_ddns_url(cfg_data['ddns_url'], cfg_data['ddns_url'])

    print(f'Saving Sync Publisher settings for {name}')
    persist_sync_publisher_config(cfg_data['sync_publisher_data'])

    print(f'Saving Object Storage Server settings for {name}')
    persist_garage_storage_config(cfg_data, KeyringService.LocalConsole)
    persist_config_data_blob(cfg_data)

    print('\n# ---------------------------------------------------------------')
    print(f'Initial configuration settings complete for {name}')


def _save_impersonation_config_data(cfg_data: Impersonation):
    """
    Helper function to save configuration data for local appliance.
    """
    impersonated_data = cfg_data['impersonated']
    impersonator_data = cfg_data['impersonator']
    impersonated_name = impersonated_data['name']

    print(f'Saving settings for {impersonator_data["name"]} to impersonate {impersonated_name}')
    _save_local_appliance_config_data(impersonated_data)
    persist_impersonator_config(impersonator_data)
    persist_garage_storage_config(impersonator_data, KeyringService.ImpersonatingConsole)

    persist_ddns_url(impersonator_data['ddns_url'], impersonated_data['ddns_url'])

    print(f'You can now restore {impersonated_name!r} database via Capsule\n')

    print('\n# ---------------------------------------------------------------')
    print('Initial configuration settings complete')


__all__ = (
    'register',
    'impersonate',
    'restore',
    'reset',
)
