from invoke import call, Context, task

from .sys import restart
from .tls import refresh_console_cert, refresh_redwood_ca
from logcabinctl.settings import (
    CONSOLE_HOME,
    IS_DRAWBRIDGE_OS,
    PYTHON,
)


@task(post=[restart])
def update(ctx: Context):
    """
    Run upgrades script(s)
    """
    print('Running console upgrade script(s)')
    if IS_DRAWBRIDGE_OS:
        ctx.sudo(f'{PYTHON} {CONSOLE_HOME}/logcabin_upgrades.py', pty=True, user='portal')
        ctx.run('update portal static', pty=True)
        ctx.run('update redwood patterns', pty=True)
    else:
        ctx.run(f'{PYTHON} {CONSOLE_HOME}/logcabin_upgrades.py', pty=True)
        ctx.run('/usr/bin/bash /opt/sap/scripts/rulesupdater.sh', pty=True)
        ctx.run('/usr/bin/bash /opt/sap/scripts/staticupdater.sh', pty=True)


@task(help={'force': 'Reload configuration files whether stale or not'})
def reload_config(ctx: Context, force=False):
    """
    Regenerate Console and Redwood configuration files
    """
    print('Regenerate Console and Redwood configuration files')
    cmd = f'{PYTHON} {CONSOLE_HOME}/manage.py reload_config'

    if force:
        cmd = f'{cmd} --force'

    if IS_DRAWBRIDGE_OS:
        ctx.sudo(cmd, pty=True, user='portal')
    else:
        ctx.run(cmd, pty=True)


@task(
    pre=[update, reload_config],
    help={'force': 'Refresh Console & Redwood CA certificates whether stale or not'},
)
def deploy(ctx: Context, force=False):  # noqa
    """
    Run upgrade script(s) and refresh webserver certificate
    """
    (call(refresh_redwood_ca, force=force),)
    (call(refresh_console_cert, force=force),)
    print('Console updated and deployed')


__all__ = (
    'deploy',
    'reload_config',
    'update',
)
