from lcconfig import ConsoleSettingsConfig
from lchttp.htmlparsing import HTMLParser
from lcrequests import LogCabinHttpSession
from pylogcabin import LogCabinAPI
from pylogcabin.exceptions import ConsoleAPIError

from invoke import Context, task


@task(help={'url': 'URL of web page to download and parse'})
def parse(ctx: Context, url: str):  # noqa
    """
    Download Page and parse into sections of interest.
    """
    header = """
# ---------------------------------------------------------
# {section} {name}
# ---------------------------------------------------------
    """
    if not url.startswith('http'):
        url = f'https://{url}'

    try:
        request = LogCabinHttpSession(url)
        response = request.get()
        if response.status_code != 200:
            print(f'Unable to retrieve URL. Status code {response.status_code}')
            return
    except Exception as e:
        print(e)
        print('An error occurred; Unable to retrieve URL')
        return

    hp = HTMLParser(response.text)

    print(header.format(section='URL:', name=url))
    print(header.format(section='Page Title:', name=hp.title()))
    print(header.format(section='Keywords & Tags', name=''))
    for kw in hp.keywords():
        print(kw)

    print(header.format(section='Asset Domains', name=''))
    for domain in hp.asset_domains():
        print(domain)

    print(header.format(section='External Domains', name=''))
    for domain in hp.external_domains():
        print(domain)


@task
def blur(ctx: Context):  # noqa
    """
    Toggle image blurring for devices in the Pumpkineers device group.
    """
    cfg = ConsoleSettingsConfig().as_typed_dict('BoxData')

    IMAGE_BLUR_CID = '449448a1-b769-611f-928d-e3df1799c6c9'

    api = LogCabinAPI(token=cfg['auth_token'])
    api.load('console', 'filterparameter')
    try:
        resp = api.post(IMAGE_BLUR_CID, subpath='toggle_active_status')
    except ConsoleAPIError as e:
        raise SystemExit(e.response.content)

    data = resp.json()

    print(f'\nImage blurring turned {data["status"].title()}\n')


__all__ = (
    'blur',
    'parse',
)
