from copy import deepcopy
from unittest import TestCase

from lcutils import NetworkInfo


# -----------------------------------------------------------------------
class TestNetworkInfo(TestCase):

    @classmethod
    def setUpClass(cls):
        super().setUpClass()
        cls.gateway_dns = 'tomcat.poweredbyclear.com'
        cls.standalone_dns = 'housecat.poweredbyclear.com'
        cls.gateway = [
            '# Network mode\n',
            'MODE="gateway"\n',
            '\n',
            '# Network interface roles\n',
            "EXTIF='eth0'\n",
            'LANIF="eth1"\n',
            'DMZIF=""\n',
            'HOTIF=""\n',
            '\n',
            '# Domain and Internet Hostname\n',
            'DEFAULT_DOMAIN="tomcat.lan"\n',
            'INTERNET_HOSTNAME="%s"\n' % cls.gateway_dns,
            '\n',
            '# Extra LANS\n',
            'EXTRALANS=""\n',
            'DEVICE=boxcar\n',
        ]
        cls.standalone = [
            '# Network mode\n',
            'MODE="trustedstandalone"\n',
            '\n',
            '# Network interface roles\n',
            "EXTIF='eth0'\n",
            'LANIF="eth1"\n',
            'DMZIF=""\n',
            'HOTIF=""\n',
            '\n',
            '# Domain and Internet Hostname\n',
            'DEFAULT_DOMAIN="tomcat.lan"\n',
            'INTERNET_HOSTNAME="%s"\n' % cls.standalone_dns,
            '\n',
            '# Extra LANS\n',
            'EXTRALANS=""\n',
            'DEVICE=boxcar\n',
        ]
        cls.ni = NetworkInfo()

    def test_get_lan_interface_standalone(self):
        self.ni.distro = lambda *x: 'clearos5'
        self.ni.network_config = lambda *x: self.standalone
        self.assertEqual(self.ni.lan_interface(), 'eth0')

    def test_get_lan_interface_gateway(self):
        self.ni.distro = lambda *x: 'clearos5'
        self.ni.network_config = lambda *x: self.gateway
        self.assertEqual(self.ni.lan_interface(), 'eth1')

    def test_get_config_param_single_quotes(self):
        """
        Make sure that params enclosed in single quotes are stripped
        """
        self.ni.network_config = lambda *x: self.gateway
        self.assertEqual(
            self.ni.get_config_param('EXTIF'), 'eth0'
        )

    def test_get_config_param_double_quotes(self):
        """
        Make sure that params enclosed in double quotes are stripped
        """
        self.ni.network_config = lambda *x: self.gateway
        self.assertEqual(
            self.ni.get_config_param('LANIF'), 'eth1',
        )

    def test_get_config_param_no_quotes(self):
        """
        Make sure that params with no enclosing quotes are accurate
        """
        self.ni.network_config = lambda *x: self.gateway
        self.assertEqual(
            self.ni.get_config_param('DEVICE'), 'boxcar'
        )

    def test_builtin_hostname(self):
        self.ni.network_config = lambda *x: self.gateway
        self.assertEqual(
            self.ni.builtin_hostname(), self.gateway_dns,
        )

        self.ni.network_config = lambda *x: self.standalone
        self.assertEqual(
            self.ni.builtin_hostname(), self.standalone_dns,
        )
