from __future__ import annotations
from collections import Counter
from lclazy import LazyLoader
import logging
import csv
from functools import lru_cache
import grp
import os
from pathlib import Path
import pwd
from typing import cast, Optional, Union
from .typehints import ArpEntry

distro = LazyLoader('distro', globals(), 'distro')
psutil = LazyLoader('psutil', globals(), 'psutil')

trans_tbl = str.maketrans({p: '' for p in ':-'})

logger = logging.getLogger(__name__)


# ---------------------------------------------------------------------------
@lru_cache(maxsize=None)
def get_vendor_list() -> dict[str, str]:
    """
    Process MAC Address Block list, in CSV format, from
    https://regauth.standards.ieee.org/standards-ra-web/pub/view.html#registries

    Whitespire has a `cron.monthly` script updating this file.
    https://standards-oui.ieee.org/oui/oui.csv
    """
    oui_list = Path(f'{os.environ.get("PERSIST_DIR")}/static/data_files/oui.csv')
    if not oui_list.exists():
        return {}

    registry = {}
    with open(oui_list, 'r', encoding='utf8') as oui:
        for row in csv.DictReader(oui, delimiter=','):
            registry[row['Assignment'].lower()] = row['Organization Name']

        return registry


# ---------------------------------------------------------------------------
def get_oui_vendor(mac: str) -> str:
    """
    Extract OUI Vendor from MAC Address
    Support "dash" and "colon" delimited MAC addresses
    """
    oui_segment = mac.translate(trans_tbl)[:6]
    vendors = get_vendor_list()
    return vendors.get(oui_segment.lower(), 'N/A')


# ----------------------------------------------------------------------
def get_arp_table(include_random=False) -> list[ArpEntry]:
    """
    Parse Arp Table into list of dicts

    IP address       HW type     Flags       HW address            Mask     Device
    192.168.1.151    0x1         0x2         1c:87:2c:ca:9c:71     *        eth1
    192.168.1.189    0x1         0x2         94:57:a5:a0:f0:f8     *        eth1
    """

    # pass names list because DictReader doesn't handle column names with spaces
    # when the delimiter is space
    names = [
        'IP address',
        'HW type',
        'Flags',
        'HW address',
        'Mask',
        'Device',
    ]
    gratuitous = frozenset(('00:00:00:00:00:00', '00-00-00-00-00-00'))
    random_mac_bits = frozenset(('2', '6', 'a', 'e'))
    incomplete_flag = '0x0'
    arp_table = []

    with open('/proc/net/arp', 'r') as arp:
        reader = csv.DictReader(arp, fieldnames=names, skipinitialspace=True, delimiter=' ')
        next(reader)  # skip first line

        for entry in reader:
            # Yes, it's actually happened that link-local IPs appear in ARP table
            if entry['IP address'].startswith('169.254.'):
                continue
            if entry['Flags'] == incomplete_flag or entry['HW address'] in gratuitous:
                continue

            entry['Randomized'] = entry['HW address'][1] in random_mac_bits
            arp_table.append(entry)

    if not include_random:
        arp_table = [entry for entry in arp_table if not entry['Randomized']]

    frequency = Counter([arp['HW address'] for arp in arp_table])
    for arp in arp_table:
        arp['mac_frequency'] = frequency[arp['HW address']]

    return cast(list[ArpEntry], arp_table)


# ---------------------------------------------------------------
def get_mac_from_arp(ip: str) -> str:
    """
    Return the MAC Address of the specified IP address from the ARP table.
    """
    ip_mac = {e['IP address']: e['HW address'] for e in get_arp_table(include_random=True)}
    try:
        return ip_mac[ip]
    except KeyError:
        pass

    return ''


# ---------------------------------------------------------------
def get_arp_entry(mac: str) -> Optional[ArpEntry]:
    """
    Get ARP Entry from the ARP table.
    """
    for ae in get_arp_table(include_random=True):
        if ae['HW address'] == mac:
            return ae

    return None


# ---------------------------------------------------------------
def get_distro() -> str:
    """Get linux distribution of host system"""

    # ('ClearOS Community', '6.7.0', 'Final')
    # ('ClearOS', '7.2.0', 'Final')
    release = distro.linux_distribution()
    os_platform = release[0].split()[0].lower()
    major_version = release[1].split('.')[0]

    if release[0].lower().startswith('centos') and release[1].startswith('5'):
        os_platform = 'clearos'
        major_version = '5'

    return f'{os_platform}{major_version}'


# ---------------------------------------------------------------
def get_process_memory(process):
    """
    Get Memory usage of a process by name / PID
    :param process: Process name or PID
    :return: Memory usage
    """

    try:
        process = int(process)
        try:
            return psutil.Process(process).memory_info().rss
        except psutil.NoSuchProcess:
            return None
    except ValueError:
        pass

    if isinstance(process, str):
        for proc in psutil.process_iter():
            try:
                pid = proc.as_dict(attrs=['pid', 'name', 'memory_info'])
                if pid['name'] == process:
                    return pid['memory_info'].rss
            except psutil.NoSuchProcess:
                pass


# ---------------------------------------------------------------------------
class GidUid:
    """
    Context Manager to specify GID / UID for a given
    operation and then revert to the previous values.
    """

    def __init__(self, gid, uid):
        self.new_gid = gid
        self.new_uid = uid
        self.current_uid = os.geteuid()
        self.current_gid = os.getegid()

    def __enter__(self):
        set_gid_uid(self.new_gid, self.new_uid)

    def __exit__(self, exc_type, exc_value, exc_traceback):
        set_gid_uid(self.current_gid, self.current_uid)


# ---------------------------------------------------------------------------
def set_gid_uid(gid: Union[int, str], uid: Union[int, str]) -> None:
    """
    Set user and group so that files written & pid files set during
    execution will retain the desired perms. Useful for running
    Django management commands.

    :param gid: Posix Group name or int - 1000, tom, root, harry
    :param uid: Posix User name int - 1000, tom, root, harry
    """
    current_uid = os.geteuid()
    current_gid = os.getegid()

    if isinstance(gid, int) and isinstance(uid, int):
        new_gid, new_uid = gid, uid
    else:
        try:
            new_gid = grp.getgrnam(str(gid)).gr_gid
            new_uid = pwd.getpwnam(str(uid)).pw_uid
        except KeyError as e:
            logger.error(f'Unable to set GID / UID: {e}')
            return

    try:
        if current_gid != new_gid:
            try:
                os.setresgid(new_gid, new_gid, -1)
            except KeyError:
                pass

        if current_uid != new_uid:
            try:
                os.setresuid(new_uid, new_uid, -1)
            except KeyError:
                pass
    except PermissionError as e:
        logger.error(f'Inadequate permissions to set GID / UID: {e}')


__all__ = (
    'GidUid',
    'get_arp_table',
    'get_mac_from_arp',
    'get_arp_entry',
    'get_distro',
    'get_oui_vendor',
    'get_process_memory',
    'set_gid_uid',
)
